package com.bxm.adsmanager.web.controller.adxadvertiserinfo;

import java.io.ByteArrayOutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.adsmanager.integration.advertiser.model.AdvertiserDto;
import com.bxm.adsmanager.integration.advertiser.service.AdShopIntegration;
import com.bxm.adsmanager.integration.utils.DateUtil;
import com.bxm.adsmanager.model.constant.CommonConstant;
import com.bxm.adsmanager.model.dao.bes.BesAdvertiserInfo;
import com.bxm.adsmanager.model.dto.bes.AuthHeader;
import com.bxm.adsmanager.model.dto.bes.AdxAdvertiserSearchDTO;
import com.bxm.adsmanager.model.dto.bes.BesBaseResponse;
import com.bxm.adsmanager.model.dto.bes.advertiser.BesAdvertiserAddDTO;
import com.bxm.adsmanager.model.dto.bes.advertiser.BesAdvertiserDTO;
import com.bxm.adsmanager.model.dto.bes.advertiser.BesAdvertiserGetDTO;
import com.bxm.adsmanager.model.dto.bes.advertiser.BesAdvertiserResultDTO;
import com.bxm.adsmanager.model.dto.bes.qualification.BesMainQualificationUpdateDTO;
import com.bxm.adsmanager.model.dto.bes.qualification.BesQualificationResultDTO;
import com.bxm.adsmanager.model.dto.bes.qualification.BesQueryQualificationInfoDTO;
import com.bxm.adsmanager.model.dto.bes.qualification.BesUploadQualificationDTO;
import com.bxm.adsmanager.model.enums.BesStatusEnum;
import com.bxm.adsmanager.model.vo.bes.BesAdvertiserInfoDetailVO;
import com.bxm.adsmanager.model.vo.bes.BesAdvertiserInfoListVO;
import com.bxm.adsmanager.service.adxadvertiserinfo.AdxAdvertiserInfoService;
import com.bxm.adsmanager.utils.BesApiUtil;
import com.bxm.adsmanager.utils.FileUtils;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.dto.ResultModel;
import com.bxm.util.dto.ValidateException;
import com.github.pagehelper.PageInfo;

import net.coobird.thumbnailator.Thumbnails;

/**
 * 百度广告主信息.</br>
 * Created by zhengwangeng on 2021/9/1
 */
@RestController
@RequestMapping("/besAdvertiserInfo")
public class BesAdvertiserInfoController extends BaseController {

    private static final Logger LOGGER = LoggerFactory.getLogger(BesAdvertiserInfoController.class);

    @Autowired
    private AdxAdvertiserInfoService adxAdvertiserInfoService;

    @Autowired
    private AuthHeader authHeader;

    @Autowired
    private AdShopIntegration adShopIntegration;


    /**
     * 根据条件查询
     *
     * @param request
     * @param response
     * @param adxAdvertiserSearchDTO
     * @return
     */
    @RequestMapping(value = "/findAll", method = RequestMethod.GET)
    public ResultModel<PageInfo> findAll(HttpServletRequest request, HttpServletResponse response,
                                                 AdxAdvertiserSearchDTO adxAdvertiserSearchDTO) {
        ResultModel<PageInfo> rs = new ResultModel<>();
        try {
            adxAdvertiserSearchDTO.setAdxAdvertiserType(BesAdvertiserInfo.ADX_ADVERTISER_TYPE_BES);
            PageInfo<BesAdvertiserInfoListVO> page = adxAdvertiserInfoService.findPage(adxAdvertiserSearchDTO);
            if (page != null) {
                rs.setSuccessed(true);
            }
            rs.setReturnValue(page);
        } catch (Exception e) {
            LOGGER.error("根据条件查找百度广告主信息出错" + e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("根据条件查找百度广告主信息出错");
        }
        return rs;
    }

    /**
     * 获取详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/get", method = RequestMethod.GET)
    public ResultModel<BesAdvertiserInfoDetailVO> get(@NotNull(message = "Id不能为空！") @RequestParam(value = "id") Integer id) throws Exception {
        ResultModel<BesAdvertiserInfoDetailVO> rs = new ResultModel<>();
        BesAdvertiserInfoDetailVO besAdvertiserInfoDetailVO = adxAdvertiserInfoService.getDetail(id);
        rs.setSuccessed(true);
        rs.setReturnValue(besAdvertiserInfoDetailVO);

        return rs;
    }

    /**
     * 获取最新的审核状态
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/getBesAuditStatus", method = RequestMethod.GET)
    public ResultModel<Integer> getBesAuditStatus(@NotNull(message = "Id不能为空！") @RequestParam(value = "id") Integer id) throws Exception {
        ResultModel<Integer> rs = new ResultModel<>();
        rs.setSuccessed(true);
        rs.setReturnValue(adxAdvertiserInfoService.getBesAuditStatus(id));

        return rs;
    }

    /**
     * 新增
     * @param dto
     * @return
     */
    @RequestMapping(value = "/add", method = RequestMethod.POST)
    public ResultModel add(@RequestBody BesAdvertiserInfo besAdvertiserInfo, HttpServletRequest request, HttpServletResponse response) throws Exception {
        //User user = getUser(request, response);
        besAdvertiserInfo.setAdxAdvertiserType(BesAdvertiserInfo.ADX_ADVERTISER_TYPE_BES);
        adxAdvertiserInfoService.add(besAdvertiserInfo);
        return ResultModelFactory.SUCCESS();
    }

    /**
     * 提交审核
     * @param dto
     * @return
     */
    @RequestMapping(value = "/audit", method = RequestMethod.POST)
    public ResultModel audit(@RequestBody BesAdvertiserInfo besAdvertiserInfo, HttpServletRequest request, HttpServletResponse response) throws Exception {
        //User user = getUser(request, response);
        BesAdvertiserInfo entity = adxAdvertiserInfoService.get(besAdvertiserInfo.getId());
        if (BesAdvertiserInfo.ADX_ADVERTISER_TYPE_BES != entity.getAdxAdvertiserType()) {
            throw new ValidateException("该接口只能提交审核百度广告主！");
        }

        List<AdvertiserDto> adShopMsgs = adShopIntegration.findAdShopMsgs(String.valueOf(besAdvertiserInfo.getAdvertiserId()));
        if (CollectionUtils.isEmpty(adShopMsgs)) {
            throw new ValidateException("没有找到广告主数据！");
        }
        AdvertiserDto advertiserDto = adShopMsgs.get(0);
        BesBaseResponse advertiserCommintResponse = commintAdvertiserToBes(besAdvertiserInfo, advertiserDto);
        if (advertiserCommintResponse.success() == false) {
            throw new ValidateException(advertiserCommintResponse.getDetailMessage());
        }
        BesBaseResponse commintQualificationInfoResponse = commintQualificationInfoToBes(besAdvertiserInfo, advertiserDto);
        if (commintQualificationInfoResponse.success() == false) {
            throw new ValidateException(advertiserCommintResponse.getDetailMessage());
        }
        besAdvertiserInfo.setAuditState(BesStatusEnum.PENDING.getLogicStatus());
        adxAdvertiserInfoService.update(besAdvertiserInfo);
        return ResultModelFactory.SUCCESS();
    }


    private BesBaseResponse commintQualificationInfoToBes(BesAdvertiserInfo besAdvertiserInfo, AdvertiserDto advertiserDto) throws Exception {
        //广告主资质查询
        BesQueryQualificationInfoDTO besQueryQualificationInfoDTO = new BesQueryQualificationInfoDTO();
        besQueryQualificationInfoDTO.setAuthHeader(authHeader);
        besQueryQualificationInfoDTO.setAdvertiserId(besAdvertiserInfo.getAdvertiserId().longValue());
        besQueryQualificationInfoDTO.setNeedLicenceImgUrl(Boolean.FALSE);

        BesQualificationResultDTO queryQualificationInfoResult = BesApiUtil.queryQualificationInfo(besQueryQualificationInfoDTO);

        //资质信息
        BesUploadQualificationDTO.APIAdvertiserLicence apiAdvertiserLicence = new BesUploadQualificationDTO.APIAdvertiserLicence();
        apiAdvertiserLicence.setType(besAdvertiserInfo.getLicenceType());
        apiAdvertiserLicence.setName(besAdvertiserInfo.getAdvertiserName());
        apiAdvertiserLicence.setNumber(advertiserDto.getBusinessLicenseNo());
        String validDate = "2030-12-31";
        if (besAdvertiserInfo.getValidDate() != null) {
            validDate = DateUtil.dateTo8String(besAdvertiserInfo.getValidDate());
        }
        apiAdvertiserLicence.setValidDate(validDate);

        //apiAdvertiserLicence.setImgDatas();
        //apiAdvertiserLicence.setImgUrls(Arrays.asList("https://bxm-news.oss-cn-hangzhou.aliyuncs.com/ad/AA4D73CE-A6B6-4C8C-918A-D6846DC5FEC5.png"));

        List<byte[]> imgDatas = new ArrayList<>(3);
        String imgUrl = advertiserDto.getBusinessLicenseImg();
        if (StringUtils.isNotBlank(imgUrl)) {
            int fileLength = FileUtils.getFileLength(imgUrl);
            //原有资质
            try(ByteArrayOutputStream outStream = new ByteArrayOutputStream()) {
                //大于2m就处理下
                if (fileLength/1024 >= 2*1024) {
                    Thumbnails.of(new URL(imgUrl)).scale(1f).outputQuality(0.5f).toOutputStream(outStream);
                } else {
                    Thumbnails.of(new URL(imgUrl)).scale(1f).toOutputStream(outStream);
                }
                byte[] bytes = outStream.toByteArray();
                imgDatas.add(bytes);
            }catch (Exception e){
                throw new ValidateException(e);
            }
        }

        //补充的资质图片地址，多个用逗号分隔
        if (StringUtils.isNotBlank(besAdvertiserInfo.getSupplementLicenseImg())) {
            String[] split = besAdvertiserInfo.getSupplementLicenseImg().split(CommonConstant.BaseCharacter.COMMA);
            for (String img : split) {
                if (StringUtils.isNotBlank(img)) {
                    byte[] bytes = IOUtils.toByteArray(new URL(img));
                    imgDatas.add(bytes);
                }
            }
        }
        if (CollectionUtils.isEmpty(imgDatas)) {
            throw new ValidateException("广告主资质图片为空，无法提交审核！");
        }
        apiAdvertiserLicence.setImgDatas(imgDatas);

        //查询结果判断
        BesQualificationResultDTO.APIAdvertiserQualificationInfo qualification = queryQualificationInfoResult.getQualification();
        if (qualification != null && qualification.getMainLicence() != null) {
            BesMainQualificationUpdateDTO request = new BesMainQualificationUpdateDTO();
            request.setAuthHeader(authHeader);

            BesMainQualificationUpdateDTO.APIAdvertiserMainQualificationUpdate apiAdvertiserMainQualificationUpdate =
                    new BesMainQualificationUpdateDTO.APIAdvertiserMainQualificationUpdate();
            apiAdvertiserMainQualificationUpdate.setAdvertiserId(besAdvertiserInfo.getAdvertiserId().longValue());
            apiAdvertiserMainQualificationUpdate.setMainLicence(apiAdvertiserLicence);

            request.setQualification(apiAdvertiserMainQualificationUpdate);
            return BesApiUtil.updateMainQualification(request);
        } else {
            BesUploadQualificationDTO.APIAdvertiserQualificationUpload apiAdvertiserQualificationUpload =
                    new BesUploadQualificationDTO.APIAdvertiserQualificationUpload();
            apiAdvertiserQualificationUpload.setAdvertiserId(besAdvertiserInfo.getAdvertiserId().longValue());
            apiAdvertiserQualificationUpload.setMainLicence(apiAdvertiserLicence);

            BesUploadQualificationDTO request = new BesUploadQualificationDTO();
            request.setAuthHeader(authHeader);
            request.setQualifications(Arrays.asList(apiAdvertiserQualificationUpload));

            return BesApiUtil.uploadQualification(request);
        }
    }


    private BesBaseResponse commintAdvertiserToBes(BesAdvertiserInfo besAdvertiserInfo, AdvertiserDto advertiserDto){
        BesAdvertiserGetDTO besAdvertiserGetDTO = new BesAdvertiserGetDTO();
        besAdvertiserGetDTO.setAdvertiserIds(Arrays.asList(besAdvertiserInfo.getAdvertiserId().longValue()));
        besAdvertiserGetDTO.setAuthHeader(authHeader);

        //新增
        BesAdvertiserDTO besAdvertiserDTO = new BesAdvertiserDTO();
        besAdvertiserDTO.setAdvertiserId(besAdvertiserInfo.getAdvertiserId().longValue());
        besAdvertiserDTO.setAdvertiserLiteName(advertiserDto.getCompany());
        besAdvertiserDTO.setAdvertiserName(besAdvertiserInfo.getAdvertiserName());
        besAdvertiserDTO.setSiteName(besAdvertiserInfo.getSiteName());
        besAdvertiserDTO.setSiteUrl(besAdvertiserInfo.getSiteUrl());
        //besAdvertiserDTO.setTelephone("18616573435");
        //besAdvertiserDTO.setAddress("西湖区文一西路767号西溪国际商务中心E幢7楼 ");
        //besAdvertiserDTO.setIsWhiteUser(0);

        BesAdvertiserAddDTO besAdvertiserAddDTO = new BesAdvertiserAddDTO();
        besAdvertiserAddDTO.setRequest(Arrays.asList(besAdvertiserDTO));
        besAdvertiserAddDTO.setAuthHeader(authHeader);

        //是否广告主已添加：是则需要修改，否则添加
        BesAdvertiserResultDTO advertiser = BesApiUtil.getAdvertiser(besAdvertiserGetDTO);
        if (advertiser.success() && CollectionUtils.isNotEmpty(advertiser.getResponse())) {
            return BesApiUtil.updateAdvertiser(besAdvertiserAddDTO);
        } else {
            return BesApiUtil.addAdvertiser(besAdvertiserAddDTO);
        }
    }
}
