package com.bxm.adsmanager.web.controller.agencychannel;

import java.sql.SQLIntegrityConstraintViolationException;
import java.util.List;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adsmanager.model.dto.OcpxRtbConfigDto;
import com.google.common.base.Preconditions;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import com.bxm.adsmanager.ecxeption.BusinessException;
import com.bxm.adsmanager.model.dao.agencychannel.AgencyChannel;
import com.bxm.adsmanager.model.dao.user.User;
import com.bxm.adsmanager.model.dto.AgencyChannelDTO;
import com.bxm.adsmanager.model.dto.AgencyChannelImportExcelDTO;
import com.bxm.adsmanager.model.dto.AgencyChannelSearchDTO;
import com.bxm.adsmanager.service.agencychannel.AgencyChannelService;
import com.bxm.adsmanager.utils.BeanValidator;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.dto.ResultModel;
import com.bxm.warcar.aspect.before.LogBefore;
import com.github.pagehelper.PageInfo;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;

/**
 * .</br>
 * Created by zhengwangeng on 2021/11/10.
 */
@RestController
@RequestMapping("/agencyChannel")
public class AgencyChannelController extends BaseController {

	private static final Logger LOGGER = LoggerFactory.getLogger(AgencyChannelController.class);

	@Autowired
	private AgencyChannelService agencyChannelService;

	/**
	 * 代理渠道添加
	 * @param agencyChannelDto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/agencyChannel/add", keyName = "代理渠道添加")
	@RequestMapping(value = "/add", method = RequestMethod.POST, produces = "application/json")
	public ResultModel add(@RequestBody AgencyChannelDTO agencyChannelDto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		try {
			//参数校验
			if (BeanValidator.validateGroup(agencyChannelDto, AgencyChannelDTO.Add.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(agencyChannelDto, AgencyChannelDTO.Add.class).getMessage());
			}
			//agencyChannelDto.setCreateTime(new Date());
			agencyChannelDto.setBelonger(user.getUsername());
			Long positionAssetSizeId = agencyChannelService.add(agencyChannelDto);

			return ResultModelFactory.SUCCESS(positionAssetSizeId);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("添加代理渠道出错" + e.getMessage(), e);
			Throwable cause = e.getCause();
			if (cause instanceof SQLIntegrityConstraintViolationException){
				return ResultModelFactory.FAIL500("该渠道号或DP链接已存在");
			}
			return ResultModelFactory.FAIL500("更新代理渠道出错");
		}
	}

	/**
	 * 代理渠道更新
	 * @param agencyChannelDto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/agencyChannel/update", keyName = "代理渠道更新")
	@RequestMapping(value = "/update", method = RequestMethod.POST, produces = "application/json")
	public ResultModel update(@RequestBody AgencyChannelDTO agencyChannelDto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		ResultModel rs = new ResultModel();
		if (agencyChannelDto == null || agencyChannelDto.getId() == null) {
			return ResultModelFactory.FAIL400("代理渠道id不能为空");
		}

		try {
			//参数校验
			if (BeanValidator.validateGroup(agencyChannelDto, AgencyChannelDTO.Update.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(agencyChannelDto, AgencyChannelDTO.Update.class).getMessage());
			}
			//agencyChannelDto.setModifyTime(new Date());
			//agencyChannelDto.setModifyUser(user.getUsername());
            Integer result = agencyChannelService.update(agencyChannelDto, user);
			rs.setReturnValue(result);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("更新代理渠道出错" + e.getMessage(), e);
			Throwable cause = e.getCause();
			if (cause instanceof SQLIntegrityConstraintViolationException){
				return ResultModelFactory.FAIL500("该渠道号或DP链接已存在");
			}
			return ResultModelFactory.FAIL500("更新代理渠道出错");
		}
		return rs;
	}

	/**
	 * 查找所有代理渠道
	 * @param searchDto
	 * @return
	 */
	@RequestMapping(value = "/findAll", method = RequestMethod.GET)
	public ResultModel<PageInfo<AgencyChannel>> findAll(HttpServletRequest request, HttpServletResponse response,
														AgencyChannelSearchDTO searchDto) {
		User user = getUser(request, response);
		ResultModel<PageInfo<AgencyChannel>> rs = new ResultModel<>();
		try {
			PageInfo<AgencyChannel> result = agencyChannelService.findAll(searchDto, user);
			rs.setReturnValue(result);
		} catch (Exception e) {
			LOGGER.error("查找代理渠道出错" + e.getMessage(), e);
			return ResultModelFactory.FAIL500("查找代理渠道出错");
		}
		return rs;
	}

	/**
	 * 删除
	 * @param id
	 * @return
	 */
	@LogBefore(operType="/agencyChannel/delete",keyName="代理渠道删除")
	@RequestMapping(value = "/delete", method = RequestMethod.POST)
	public ResultModel delete(HttpServletRequest request, HttpServletResponse response,
							  @RequestParam(required = true, name = "id") Long id) {
		User user = getUser(request, response);
		ResultModel resultModel = new ResultModel<>();
		try {
			resultModel.setSuccessed(agencyChannelService.delete(id, user));
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e){
			return ResultModelFactory.FAIL500("删除失败");
		}
		return resultModel;
	}

	/**
	 * 导入代理渠道
	 * @param request
	 * @param response
	 * @param file
	 * @return
	 */
	@LogBefore(operType = "/agencyChannel/import", keyName = "导入代理渠道")
	@RequestMapping(value = "/import", method = RequestMethod.POST)
	public ResultModel<String> importData(HttpServletRequest request, HttpServletResponse response,
										  @RequestParam("file") MultipartFile file) {
		User user = getUser(request, response);
		try {
			ImportParams importParams = new ImportParams();
			List<AgencyChannelImportExcelDTO> list = ExcelImportUtil.importExcel(file.getInputStream(), AgencyChannelImportExcelDTO.class, importParams);
			if (CollectionUtils.isEmpty(list)) {
				return ResultModelFactory.FAIL500("导入数据为空");
			}
			agencyChannelService.importData(list, user);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("导入代理渠道数据出错" + e.getMessage(), e);
			Throwable cause = e.getCause();
			if (cause instanceof SQLIntegrityConstraintViolationException){
				return ResultModelFactory.FAIL500("文件中已有渠道收入数据录入");
			}
			return ResultModelFactory.FAIL500("导入代理渠道数据出错");
		}
		return ResultModelFactory.SUCCESS();
	}

    /**
     * ocpx-rtb配置
     * @return
     */
    @LogBefore(operType = "/agencyChannel/ocpxConfig/update", keyName = "ocpx-rtb配置编辑")
    @RequestMapping(value = "/ocpxConfig/update", method = RequestMethod.POST)
    public ResultModel<String> updateOcpxConfig(@Validated @RequestBody OcpxRtbConfigDto ocpxRtbConfigDto) {
         checkUpdateOcpxConfig(ocpxRtbConfigDto);
        try {
            String url = agencyChannelService.updateOcpxConfig(ocpxRtbConfigDto);
            return ResultModelFactory.SUCCESS(url);
        } catch (BusinessException e) {
            return ResultModelFactory.FAIL500(e.getMessage());
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
            return ResultModelFactory.FAIL500("系统错误");
        }
    }

    private void checkUpdateOcpxConfig(OcpxRtbConfigDto ocpxRtbConfigDto) {
        Double rate = ocpxRtbConfigDto.getRate();
        if (Objects.nonNull(ocpxRtbConfigDto.getStrategyType()) && ocpxRtbConfigDto.getStrategyType() == 2) {
            Preconditions.checkArgument(Objects.nonNull(rate) && rate > 0d && rate <= 100d, "rate有误");
        }
        if (ocpxRtbConfigDto.getTargetType() == 2) {
            Preconditions.checkArgument(StringUtils.isNotBlank(ocpxRtbConfigDto.getTargetTwoRtb()), "targetTwoRtb不能为空");
            Preconditions.checkArgument(StringUtils.isNotBlank(ocpxRtbConfigDto.getRelateTwoRtb()), "relateTwoRtb不能为空");
        }
    }

//
//    public static void main(String[] args) {
//        OcpxRtbConfigDto ocpxRtbConfigDto = new OcpxRtbConfigDto();
//        ocpxRtbConfigDto.setId(21L);
//        ocpxRtbConfigDto.setRate(99d);
//        ocpxRtbConfigDto.setStrategyType(2);
//        ocpxRtbConfigDto.setRtbMedia("domob");
//        ocpxRtbConfigDto.setRtbAdvertiser("taobao");
//        ocpxRtbConfigDto.setTargetType(2);
//        ocpxRtbConfigDto.setTargetOneRtb("7");
//        ocpxRtbConfigDto.setTargetTwoRtb("8");
//        ocpxRtbConfigDto.setRelateOneRtb("12");
//        ocpxRtbConfigDto.setRelateTwoRtb("14");
//
//        String s = "[{\"field\": \"advertiserSpaceId\", \"value\": \"7778888\"}, {\"field\": \"taskId\", \"value\": \"1234\"}]";
//        ocpxRtbConfigDto.setCustomUrlFields(s);
//        System.out.println(JSONObject.toJSONString(ocpxRtbConfigDto));
//    }
}