package com.bxm.adsmanager.web.controller.agencychannel;

import java.sql.SQLIntegrityConstraintViolationException;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import com.bxm.adsmanager.ecxeption.BusinessException;
import com.bxm.adsmanager.model.dao.agencychannel.AgencyChannelData;
import com.bxm.adsmanager.model.dao.user.User;
import com.bxm.adsmanager.model.dto.AgencyChannelDataBatchUpdateDTO;
import com.bxm.adsmanager.model.dto.AgencyChannelDataDTO;
import com.bxm.adsmanager.model.dto.AgencyChannelDataImportExcelDTO;
import com.bxm.adsmanager.model.dto.AgencyChannelDataSearchDTO;
import com.bxm.adsmanager.model.dto.AgencyChannelDataSyncDTO;
import com.bxm.adsmanager.service.agencychannel.AgencyChannelDataService;
import com.bxm.adsmanager.utils.BeanValidator;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.dto.ResultModel;
import com.bxm.warcar.aspect.before.LogBefore;
import com.github.pagehelper.PageInfo;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.result.ExcelImportResult;

/**
 * 代理渠道数据数据.</br>
 * Created by zhengwangeng on 2021/11/10.
 */
@RestController
@RequestMapping("/agencyChannelData")
public class AgencyChannelDataController extends BaseController {

	private static final Logger LOGGER = LoggerFactory.getLogger(AgencyChannelDataController.class);

	@Autowired
	private AgencyChannelDataService agencyChannelDataService;

	/**
	 * 代理渠道数据添加
	 * @param agencyChannelDataDto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/agencyChannelData/add", keyName = "代理渠道数据添加")
	@RequestMapping(value = "/add", method = RequestMethod.POST, produces = "application/json")
	public ResultModel add(@RequestBody AgencyChannelDataDTO agencyChannelDataDto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		try {
			//参数校验
			if (BeanValidator.validateGroup(agencyChannelDataDto, AgencyChannelDataDTO.Add.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(agencyChannelDataDto, AgencyChannelDataDTO.Add.class).getMessage());
			}
			agencyChannelDataDto.setBelonger(user.getUsername());
			Long positionAssetSizeId = agencyChannelDataService.add(agencyChannelDataDto, user);

			return ResultModelFactory.SUCCESS(positionAssetSizeId);
		} catch (Exception e) {
			LOGGER.error("添加代理渠道数据出错" + e.getMessage(), e);
			Throwable cause = e.getCause();
			if (cause instanceof SQLIntegrityConstraintViolationException){
				return ResultModelFactory.FAIL500(agencyChannelDataDto.getChannelNo() + "收入数据已录入");
			}
			return ResultModelFactory.FAIL500("添加代理渠道数据出错");
		}
	}

	/**
	 * 代理渠道数据更新
	 * @param agencyChannelDataDto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/agencyChannelData/update", keyName = "代理渠道数据更新")
	@RequestMapping(value = "/update", method = RequestMethod.POST, produces = "application/json")
	public ResultModel update(@RequestBody AgencyChannelDataDTO agencyChannelDataDto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		ResultModel rs = new ResultModel();
		if (agencyChannelDataDto == null || agencyChannelDataDto.getId() == null) {
			return ResultModelFactory.FAIL400("代理渠道数据id不能为空");
		}

		try {
			//参数校验
			if (BeanValidator.validateGroup(agencyChannelDataDto, AgencyChannelDataDTO.Update.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(agencyChannelDataDto, AgencyChannelDataDTO.Update.class).getMessage());
			}
			//agencyChannelDataDto.setModifyTime(new Date());
			//agencyChannelDataDto.setModifyUser(user.getUsername());
            Integer result = agencyChannelDataService.update(agencyChannelDataDto, user);
			rs.setReturnValue(result);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("更新代理渠道数据出错" + e.getMessage(), e);
			Throwable cause = e.getCause();
			if (cause instanceof SQLIntegrityConstraintViolationException){
				return ResultModelFactory.FAIL500(agencyChannelDataDto.getChannelNo() + "收入数据已录入");
			}
			return ResultModelFactory.FAIL500("更新代理渠道数据出错");
		}
		return rs;
	}

	/**
	 * 代理渠道数据批量更新
	 * @param dto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/agencyChannelData/batchUpdate", keyName = "代理渠道数据批量更新")
	@RequestMapping(value = "/batchUpdate", method = RequestMethod.POST, produces = "application/json")
	public ResultModel batchUpdate(@RequestBody AgencyChannelDataBatchUpdateDTO dto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		ResultModel rs = new ResultModel();

		try {
			//参数校验
			if (BeanValidator.validateGroup(dto, AgencyChannelDataDTO.Update.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(dto, AgencyChannelDataDTO.Update.class).getMessage());
			}
			if (dto.getValidClickPrice() == null && dto.getQualityFactor() == null && StringUtils.isBlank(dto.getRemark())){
				return ResultModelFactory.FAIL400("批量修改的值不能都为空");
			}
			rs.setReturnValue(agencyChannelDataService.batchUpdate(dto, user));
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("更新代理渠道数据出错" + e.getMessage(), e);
			return ResultModelFactory.FAIL500("更新代理渠道数据出错");
		}
		return rs;
	}

	/**
	 * 导入代理渠道数据
	 * @param request
	 * @param response
	 * @param file
	 * @return
	 */
	@LogBefore(operType = "/agencyChannelData/import", keyName = "导入代理渠道数据")
	@RequestMapping(value = "/import", method = RequestMethod.POST)
	public ResultModel importData(HttpServletRequest request, HttpServletResponse response,
										  @RequestParam("file") MultipartFile file) {
		User user = getUser(request, response);
		ResultModel rs = new ResultModel();
		try {
			ImportParams importParams = new ImportParams();
			importParams.setNeedVerfiy(true);
			ExcelImportResult<AgencyChannelDataImportExcelDTO> importResult = ExcelImportUtil.importExcelMore(file.getInputStream(),
					AgencyChannelDataImportExcelDTO.class, importParams);
			List<AgencyChannelDataImportExcelDTO> successList = importResult.getList();
			List<AgencyChannelDataImportExcelDTO> failList = importResult.getFailList();
			if (CollectionUtils.isEmpty(successList) && CollectionUtils.isEmpty(failList)) {
				return ResultModelFactory.FAIL500("导入失败数据为空。" );
			}
			rs.setReturnValue(agencyChannelDataService.importData(successList, user,failList));
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("导入代理渠道数据出错" + e.getMessage(), e);
			Throwable cause = e.getCause();
			if (cause instanceof SQLIntegrityConstraintViolationException){
				return ResultModelFactory.FAIL500("文件中已有渠道收入数据录入");
			}
			return ResultModelFactory.FAIL500("导入代理渠道数据出错");
		}
		rs.setSuccessed(Boolean.TRUE);
		return rs;
	}

	/**
	 * 查找所有代理渠道数据
	 * @param searchDto
	 * @return
	 */
	@RequestMapping(value = "/findAll", method = RequestMethod.GET)
	public ResultModel<PageInfo<AgencyChannelData>> findAll(HttpServletRequest request, HttpServletResponse response,
														AgencyChannelDataSearchDTO searchDto) {
		User user = getUser(request, response);
		ResultModel<PageInfo<AgencyChannelData>> rs = new ResultModel<>();
		try {
			PageInfo<AgencyChannelData> result = agencyChannelDataService.findAll(searchDto, user);
			rs.setReturnValue(result);
		} catch (Exception e) {
			LOGGER.error("查找代理渠道数据出错" + e.getMessage(), e);
			return ResultModelFactory.FAIL500("查找代理渠道数据出错");
		}
		return rs;
	}

	/**
	 * 删除
	 * @param ids
	 * @return
	 */
	@LogBefore(operType="/agencyChannelData/delete",keyName="代理渠道数据删除")
	@RequestMapping(value = "/delete", method = RequestMethod.POST)
	public ResultModel delete(HttpServletRequest request, HttpServletResponse response,
							  @RequestParam(required = true, name = "ids") List<Long> ids) {
		User user = getUser(request, response);
		ResultModel resultModel = new ResultModel<>();
		try {
			resultModel.setReturnValue(agencyChannelDataService.delete(ids, user));
			resultModel.setSuccessed(Boolean.TRUE);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e){
			return ResultModelFactory.FAIL500("删除失败");
		}
		return resultModel;
	}

	/**
	 * 推送数据到CMS后台
	 * @param datetime
	 * @return
	 */
	@LogBefore(operType="/agencyChannelData/pushDataToCms",keyName="推送数据到CMS后台")
	@RequestMapping(value = "/pushDataToCms", method = RequestMethod.POST)
	public ResultModel pushDataToCms(HttpServletRequest request, HttpServletResponse response,
									 @Validated @RequestBody AgencyChannelDataSyncDTO dto) {
		User user = getUser(request, response);
		ResultModel resultModel = new ResultModel<>();
		try {
			resultModel.setSuccessed(agencyChannelDataService.pushDataToCms(dto, user));
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e){
			LOGGER.error("推送数据到CMS后台异常："+ ExceptionUtils.getStackTrace(e));
			return ResultModelFactory.FAIL500("推送数据到CMS后台异常");
		}
		return resultModel;
	}

}