package com.bxm.adsmanager.web.controller.agencychannel;

import java.sql.SQLIntegrityConstraintViolationException;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import com.bxm.adsmanager.ecxeption.BusinessException;
import com.bxm.adsmanager.model.dao.user.User;
import com.bxm.adsmanager.model.dto.ProxyChannelDataBatchUpdateDTO;
import com.bxm.adsmanager.model.dto.ProxyChannelDataDTO;
import com.bxm.adsmanager.model.dto.ProxyChannelDataImportExcelDTO;
import com.bxm.adsmanager.model.dto.ProxyChannelDataSearchDTO;
import com.bxm.adsmanager.model.enums.RoleEnum;
import com.bxm.adsmanager.model.vo.ProxyChannelDataVO;
import com.bxm.adsmanager.service.agencychannel.ProxyChannelDataService;
import com.bxm.adsmanager.utils.BeanValidator;
import com.bxm.adsmanager.utils.ExcelUtil;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.DateUtil;
import com.bxm.util.dto.ResultModel;
import com.bxm.warcar.aspect.before.LogBefore;
import com.github.pagehelper.PageInfo;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.result.ExcelImportResult;

/**
 * 代理数据（下游）.</br>
 * Created by zhengwangeng on 2022/6/24.
 */
@RestController
@RequestMapping("/proxyChannelData")
public class ProxyChannelDataController extends BaseController {

	private static final Logger LOGGER = LoggerFactory.getLogger(ProxyChannelDataController.class);

	@Autowired
	private ProxyChannelDataService proxyChannelDataService;

	/**
	 * 代理数据（下游）更新
	 * @param proxyChannelDataDto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/proxyChannelData/update", keyName = "代理数据（下游）更新")
	@RequestMapping(value = "/update", method = RequestMethod.POST, produces = "application/json")
	public ResultModel update(@RequestBody ProxyChannelDataDTO proxyChannelDataDto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		ResultModel rs = new ResultModel();
		testFillRole(user);
		try {
			//参数校验
			if (BeanValidator.validateGroup(proxyChannelDataDto, ProxyChannelDataDTO.Update.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(proxyChannelDataDto, ProxyChannelDataDTO.Update.class).getMessage());
			}
            proxyChannelDataService.update(proxyChannelDataDto, user);
			rs.setReturnValue(Boolean.TRUE);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("更新代理数据（下游）出错" + e.getMessage(), e);
			return ResultModelFactory.FAIL500("更新代理数据（下游）出错");
		}
		return rs;
	}


	/**
	 * 代理数据（下游）清空
	 * @param proxyChannelDataDto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/proxyChannelData/empty", keyName = "代理数据（下游）清空")
	@RequestMapping(value = "/empty", method = RequestMethod.POST, produces = "application/json")
	public ResultModel empty(@RequestBody ProxyChannelDataDTO proxyChannelDataDto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		ResultModel rs = new ResultModel();
		testFillRole(user);
		try {
			//参数校验
			if (BeanValidator.validateGroup(proxyChannelDataDto, ProxyChannelDataDTO.Empty.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(proxyChannelDataDto, ProxyChannelDataDTO.Empty.class).getMessage());
			}
			for (Long id : proxyChannelDataDto.getIds()) {
				proxyChannelDataDto.setId(id);
				proxyChannelDataService.update(proxyChannelDataDto, user);
			}
			rs.setReturnValue(Boolean.TRUE);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("清空代理数据（下游）出错" + e.getMessage(), e);
			return ResultModelFactory.FAIL500("清空代理数据（下游）出错");
		}
		return rs;
	}

	/**
	 * 代理数据（下游）批量更新
	 * @param dto
	 * @param request
	 * @param response
	 * @return
	 */
	@LogBefore(operType = "/proxyChannelData/batchUpdate", keyName = "代理数据（下游）批量更新")
	@RequestMapping(value = "/batchUpdate", method = RequestMethod.POST, produces = "application/json")
	public ResultModel batchUpdate(@RequestBody ProxyChannelDataBatchUpdateDTO dto, HttpServletRequest request, HttpServletResponse response) {
		User user = getUser(request, response);
		ResultModel rs = new ResultModel();
		testFillRole(user);
		try {
			//参数校验
			if (BeanValidator.validateGroup(dto, ProxyChannelDataBatchUpdateDTO.Update.class) != null) {
				return ResultModelFactory.FAIL400(BeanValidator.validateGroup(dto, ProxyChannelDataBatchUpdateDTO.Update.class).getMessage());
			}
			if (dto.getDs() == null && dto.getDsPrice() == null ){
				return ResultModelFactory.FAIL400("批量修改的值不能都为空");
			}
			rs.setReturnValue(proxyChannelDataService.batchUpdate(dto, user));
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("更新代理数据（下游）出错" + e.getMessage(), e);
			return ResultModelFactory.FAIL500("更新代理数据（下游）出错");
		}
		return rs;
	}

	/**
	 * 导入代理数据（下游）
	 * @param request
	 * @param response
	 * @param file
	 * @return
	 */
	@LogBefore(operType = "/proxyChannelData/import", keyName = "导入代理数据（下游）")
	@RequestMapping(value = "/import", method = RequestMethod.POST)
	public ResultModel<String> importData(HttpServletRequest request, HttpServletResponse response,
										  @RequestParam("file") MultipartFile file) {
		User user = getUser(request, response);
		try {
			ImportParams importParams = new ImportParams();
			importParams.setNeedVerfiy(true);
			ExcelImportResult<ProxyChannelDataImportExcelDTO> importResult = ExcelImportUtil.importExcelMore(file.getInputStream(),
					ProxyChannelDataImportExcelDTO.class, importParams);
			List<ProxyChannelDataImportExcelDTO> successList = importResult.getList();
			List<ProxyChannelDataImportExcelDTO> failList = importResult.getFailList();
			if (CollectionUtils.isEmpty(successList) && CollectionUtils.isEmpty(failList)) {
				return ResultModelFactory.FAIL500("导入数据失败：数据为空。" );
			}
			proxyChannelDataService.importData(successList, user,failList);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("导入代理数据（下游）出错" + e.getMessage(), e);
			Throwable cause = e.getCause();
			if (cause instanceof SQLIntegrityConstraintViolationException){
				return ResultModelFactory.FAIL500("文件中已有渠道收入数据录入");
			}
			return ResultModelFactory.FAIL500("导入代理数据（下游）出错");
		}
		return ResultModelFactory.SUCCESS();
	}

	/**
	 * 查找所有代理数据（下游）
	 * @param searchDto
	 * @return
	 */
	@RequestMapping(value = "/findAll", method = RequestMethod.GET)
	public ResultModel<PageInfo<ProxyChannelDataVO>> findAll(HttpServletRequest request, HttpServletResponse response,
															 ProxyChannelDataSearchDTO searchDto) {
		//时间校验
		checkDate(searchDto.getStartTime(), searchDto.getEndTime(), 31);
		User user = getUser(request, response);
		ResultModel<PageInfo<ProxyChannelDataVO>> rs = new ResultModel<>();
		try {
			testFillRole(user);
			PageInfo<ProxyChannelDataVO> result = proxyChannelDataService.findAll(searchDto, user);
			rs.setReturnValue(result);
		} catch (BusinessException e) {
			return ResultModelFactory.FAIL500(e.getMessage());
		} catch (Exception e) {
			LOGGER.error("查找代理数据（下游）出错" + e.getMessage(), e);
			return ResultModelFactory.FAIL500("查找代理数据（下游）出错");
		}
		return rs;
	}

	/**
	 * 本地测试时可跳过接口的权限配置
	 * @param user
	 */
	private void testFillRole(User user) {
		//ArrayList<String> roleCodes = new ArrayList<>();
		//roleCodes.add(RoleEnum.CHAN_PIN_LEADER.getCode());
		//user.setRoleCodes(roleCodes);
	}

	private void checkDate(String startTime, String endTime, int days) {
		Date sdate = DateUtil.convertStr2Date(startTime);
		Date edate = DateUtil.convertStr2Date(endTime);
		sdate = DateUtil.increaseDate(sdate, days);
		if (sdate.getTime() < edate.getTime()) {
			throw new BusinessException("查询时间不能超过 "+ days +" 天");
		}
	}


	/**
	 * 获取当前条件的数据，并返回excel模板数据（只包含渠道收益数据，不包括代理下游数据）
	 * @param searchDto
	 * @return
	 */
	@RequestMapping(value = "/template",method = RequestMethod.GET)
	public void export(HttpServletRequest request, HttpServletResponse response,
															 ProxyChannelDataSearchDTO searchDto) {
		//时间校验
		checkDate(searchDto.getStartTime(), searchDto.getEndTime(), 7);
		try {
			User user = getUser(request, response);
			List<ProxyChannelDataVO> exportList = proxyChannelDataService.getList(searchDto, user);
			if (CollectionUtils.isEmpty(exportList)){
				exportList = Collections.emptyList();
			}
			for (ProxyChannelDataVO proxyChannelDataVO : exportList) {
				proxyChannelDataVO.setDsCpsBrokerage(null);
				proxyChannelDataVO.setDsIncome(null);
				proxyChannelDataVO.setDsPrice(null);
				proxyChannelDataVO.setDsValidClick(null);
			}
			ExcelUtil.exportExcel(exportList, null, null, ProxyChannelDataVO.class, "代理数据（下游）记录导出.xls", response);
		} catch (Exception e) {
			e.printStackTrace();
			LOGGER.error("导出代理数据（下游）出错" + e.getMessage(), e);
		}
	}

}