package com.bxm.adsmanager.web.controller.audit;

import java.util.*;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import cn.afterturn.easypoi.excel.annotation.Excel;
import cn.afterturn.easypoi.excel.annotation.ExcelCollection;
import com.bxm.adsmanager.dal.mapper.advertiser.ext.TblAdQualifyMapperExt;
import com.bxm.adsmanager.model.dao.advertiser.TblAdQualify;
import com.bxm.adsmanager.model.vo.AdvertiserImgExcelVo;
import com.bxm.adsmanager.service.audit.AdvertiserAuditService;
import com.bxm.adsmanager.utils.ExcelUtil;
import com.bxm.warcar.utils.Constants;
import lombok.AllArgsConstructor;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.adsmanager.dal.mapper.adkeeper.AdTicketTagConfMapper;
import com.bxm.adsmanager.ecxeption.BusinessException;
import com.bxm.adsmanager.integration.advertiser.model.AdvertiserAuditVo;
import com.bxm.adsmanager.integration.advertiser.model.AdvertiserDto;
import com.bxm.adsmanager.integration.advertiser.model.AdvertiserMainInfoVo;
import com.bxm.adsmanager.integration.advertiser.service.AdShopIntegration;
import com.bxm.adsmanager.model.constant.CommonConstant;
import com.bxm.adsmanager.model.dao.user.User;
import com.bxm.adsmanager.model.dto.AdTicketAuditSearchDTO;
import com.bxm.adsmanager.model.vo.AdTicketAuditVo;
import com.bxm.adsmanager.model.vo.AdvertiserQualifyInfoVo;
import com.bxm.adsmanager.service.audit.AdAuditService;
import com.bxm.adsmanager.service.audit.impl.AdAuditServiceImpl;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.dto.ResultModel;
import com.bxm.warcar.aspect.before.LogBefore;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.cache.impls.redis.JedisUpdater;
import com.github.pagehelper.PageInfo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @author tangx
 * @date 2020/6/17 15:52
 */
@Slf4j
@RestController
@RequestMapping("/adAudit")
public class AdAuditController extends BaseController {

    @Autowired
    private AdAuditService adAuditService;
    @Autowired
    private AdvertiserAuditService advertiserAuditService;

    @Autowired
    private AdShopIntegration adShopIntegration;

    @Autowired
    private JedisFetcher jedisFetcher;

    @Autowired
    private JedisUpdater jedisUpdater;

    @Autowired
    private AdTicketTagConfMapper adTicketTagConfMapper;
    @Autowired
    private TblAdQualifyMapperExt adQualifyMapperExt;

    private static final ReentrantLock EXPORT_IMG_LOCK = new ReentrantLock();

    public static final String FORMAT_EXCEL = "excel";

    public static final String FORMAT_IMG = "img";




    /**
     * 广告主审核列表
     *
     * @param pageNum
     * @param pageSize
     * @param accountType        广告主类型
     * @param advertiserIdOrName id或姓名
     * @param advertiserStatus   广告主主体状态
     * @param qualifyStatus      资质状态
     * @return com.bxm.util.dto.ResultModel
     * @author tangx
     * @DATE 2020/6/17 16:02
     */
    @GetMapping("/listAdvertiserAudit")
    public ResultModel listAdvertiserAudit(@RequestParam(name = "pageNum", defaultValue = "1") Integer pageNum,
                                           @RequestParam(name = "pageSize", defaultValue = "20") Integer pageSize,
                                           @RequestParam(name = "accountType", required = false) Byte accountType,
                                           @RequestParam(name = "areaType", required = false) Integer areaType,
                                           @RequestParam(name = "advertiserIdOrName", required = false) String advertiserIdOrName,
                                           @RequestParam(name = "advertiserStatus", required = false) Byte advertiserStatus,
                                           @RequestParam(name = "qualifyStatus", required = false) Integer qualifyStatus,
                                           @RequestParam(name = "expireStatus",required = false) Integer expireStatus) {

        PageInfo<AdvertiserAuditVo> pageInfo = advertiserAuditService.listAdvertiserAudit(pageNum, pageSize, accountType, advertiserIdOrName, advertiserStatus, qualifyStatus,expireStatus,areaType);
        return ResultModelFactory.SUCCESS(pageInfo);
    }

    /**
     * 广告主主体审核
     *
     * @param id
     * @param isPass
     * @param refuseReason
     * @return
     */
    @LogBefore(operType = "/adAudit/advertiserAudit", keyName = "广告主主体审核")
    @PostMapping("/advertiserAudit")
    public ResultModel<Boolean> advertiserAudit(@RequestParam("id") Integer id,
                                                @RequestParam("isPass") Integer isPass,
                                                @RequestParam(value = "refuseReason", required = false) String refuseReason,
                                                @RequestParam(value = "refuseRemark", required = false) String refuseRemark) {
        return adShopIntegration.advertiserAudit(id, isPass, refuseReason, refuseRemark);
    }

    /**
     * 获取广告主主体信息
     *
     * @param id
     * @return
     */
    @GetMapping("/getAdvertiserMainInfo")
    public ResultModel<AdvertiserMainInfoVo> getAdvertiserMainInfo(@RequestParam("id") Integer id) {
        AdvertiserMainInfoVo advertiserMainInfo = adShopIntegration.getAdvertiserMainInfo(id);
        return ResultModelFactory.SUCCESS(advertiserMainInfo);
    }

    /**
     * 资质详细信息
     * @param qualityId
     * @return
     */
    @GetMapping("/getQualifyInfo")
    public ResultModel<AdvertiserQualifyInfoVo> getQualifyInfo(@RequestParam("qualityId") Integer qualityId) {
        AdvertiserQualifyInfoVo vo = advertiserAuditService.getQualifyInfo(qualityId);
        return ResultModelFactory.SUCCESS(vo);
    }

    /**
     * 资质审核列表
     * @param pageNum
     * @param pageSize
     * @param advertiserId
     * @param auditStatus
     * @return
     */
    @GetMapping("/getQualifyList")
    public ResultModel<PageInfo<AdvertiserQualifyInfoVo>> getQualifyList(@RequestParam(name = "pageNum", defaultValue = "1") Integer pageNum,
                                                                         @RequestParam(name = "pageSize", defaultValue = "20") Integer pageSize,
                                                                         @RequestParam("advertiserId") Integer advertiserId,
                                                                         @RequestParam(value = "auditStatus", required = false) Integer auditStatus,
                                                                         @RequestParam(value = "content", required = false) String content) {
        PageInfo<AdvertiserQualifyInfoVo> pageInfo = advertiserAuditService.getQualifyList(pageNum, pageSize, advertiserId, auditStatus, content);
        return ResultModelFactory.SUCCESS(pageInfo);
    }

    /**
     * 修改资质备注
     *
     * @param qualifyId
     * @param remark
     * @return
     */
    @PostMapping("/updateQualifyRemark")
    public ResultModel<Boolean> updateQualifyRemark(@RequestParam("qualifyId") Integer qualifyId,
                                                    @RequestParam(value = "remark", required = false) String remark) {
        advertiserAuditService.updateQualifyRemark(qualifyId, remark);
        return ResultModelFactory.SUCCESS();
    }


    /**
     * 审核后台-广告审核列表
     *
     * @param adTicketAuditSearchDTO
     * @return
     */
    @RequestMapping(value = "/getWaitCheckList", method = RequestMethod.GET)
    public ResultModel<PageInfo<AdTicketAuditVo>> getTicketAuditList(AdTicketAuditSearchDTO adTicketAuditSearchDTO) {
        ResultModel<PageInfo<AdTicketAuditVo>> resultModel = new ResultModel<>();
        try {
            // 备注搜索的ticketId
            List<Long> targetRemarkTickets = getTargetRemarkTickets(adTicketAuditSearchDTO.getRemark());
            // 标签搜索的tocetId
            if (null != adTicketAuditSearchDTO.getTagCode()) {
                List<Long> targetTagTicketIds = adTicketTagConfMapper.findTicketIdByTagCode(adTicketAuditSearchDTO.getTagCode());
                if (targetRemarkTickets == null) {
                    targetRemarkTickets = new ArrayList<>();
                    targetRemarkTickets.addAll(targetTagTicketIds);
                } else {
                    // 取交集
                    targetRemarkTickets.retainAll(targetTagTicketIds);
                }
            }
            if (StringUtils.isNotBlank(adTicketAuditSearchDTO.getStartTime()) && StringUtils.isNotBlank(adTicketAuditSearchDTO.getEndTime())) {
                adTicketAuditSearchDTO.setStartTime(adTicketAuditSearchDTO.getStartTime() + " 00:00:00");
                adTicketAuditSearchDTO.setEndTime(adTicketAuditSearchDTO.getEndTime() + " 23:59:59");
            }
            adTicketAuditSearchDTO.setTicketIdList(targetRemarkTickets);
            PageInfo<AdTicketAuditVo> page = adAuditService.getTicketAuditList(adTicketAuditSearchDTO);
            resultModel.setReturnValue(page);
        } catch (Exception e) {
            log.error("查询待审核券列表信息出错" + e.getMessage(), e);
            return ResultModelFactory.FAIL500("查询待审核券列表信息出错");
        }
        return resultModel;
    }

    /**
     * 更新备注
     *
     * @param ticketId
     * @param remark
     * @return
     */
    @PostMapping(value = "/updateTicketRemark")
    public ResultModel<Boolean> updateTicketRemark(@RequestParam("ticketId") Integer ticketId,
                                                   @RequestParam(value = "remark", required = false) String remark) {
        try {
            if (null == remark) {
                remark = "";
            }
            adAuditService.updateTicketRemark(ticketId, remark);
        } catch (Exception e) {
            return ResultModelFactory.FAIL500("备注更新失败");
        }
        return ResultModelFactory.SUCCESS();
    }

    /**
     * 修改落地页截图
     *
     * @param ticketId
     * @param url
     * @return
     */
    @PostMapping("/updateLoadingPageImg")
    public ResultModel updateLoadingPageImg(@RequestParam("ticketId") Integer ticketId, @RequestParam("url") String url) {
        if (StringUtils.isNotBlank(url)) {
            jedisUpdater.hupdate(AdAuditServiceImpl.loadingPageScreenShotUrl(), ticketId.toString(), url);
        }
        return ResultModelFactory.SUCCESS();
    }


    /**
     * 查出搜索条件输入备注 的ticketId
     *
     * @param remark
     */
    public List<Long> getTargetRemarkTickets(String remark) {
        if (StringUtils.isBlank(remark)) {
            return null;
        }
        List<Long> ticketIds = new ArrayList<>();
        Map<String, String> allRemark = jedisFetcher.hfetchall(AdAuditServiceImpl.ticketRemark(), String.class);
        allRemark.forEach((k, v) -> {
            if (v.contains(remark)) {
                ticketIds.add(Long.valueOf(k));
            }
        });
        return ticketIds;
    }

    /**
     * 素材审核
     *
     * @param id
     * @param ids
     * @param auditStatus
     * @param refuseReason
     * @param ticketType
     * @return
     */
    @LogBefore(operType = "/adAudit/assetsAudit", keyName = "素材审核")
    @PostMapping("/assetsAudit")
    public ResultModel<Boolean> asstesAudit(@RequestParam(value = "id", required = false) Integer id,
                                            @RequestParam(value = "ids", required = false) List<Long> ids,
                                            @RequestParam(value = "auditStatus") Short auditStatus,
                                            @RequestParam(value = "refuseReason", required = false) String refuseReason,
                                            @RequestParam(value = "ticketType", required = false) Integer ticketType,
                                            HttpServletRequest request, HttpServletResponse response) {
        try {
            User user = getUser(request, response);
            adAuditService.asstesAudit(id, ids, auditStatus, refuseReason, ticketType, CommonConstant.AssetCheckStatus.CHECK_STATUS_PASS, user.getUsername());
        } catch (Exception e) {
            log.info(e.getMessage(), e);
            return ResultModelFactory.FAIL500(e.getMessage());
        }
        return ResultModelFactory.SUCCESS();
    }

    @GetMapping("/exportQualityImgs")
    public ResultModel<String> exportQuality(@RequestParam(value = "advertiserIds", required = false) String advertiserIds,
                                             @RequestParam(value = "format", required = false, defaultValue = "excel") String format,
                                             HttpServletResponse response) {
        if (StringUtils.isBlank(advertiserIds)) {
            format = FORMAT_EXCEL;
        }

        // 获取锁
        boolean tryLock = false;
        try {
            tryLock = EXPORT_IMG_LOCK.tryLock();
            if (!tryLock) {
                return null;
            }
            if (Objects.equals(format, FORMAT_EXCEL)) {
                advertiserAuditService.exportImgsUseExcel(advertiserIds, response);
            }
            if (Objects.equals(format, FORMAT_IMG)) {
                advertiserAuditService.exportImgsUseZip(advertiserIds, response);
            }
            return ResultModelFactory.SUCCESS();

        } catch (NumberFormatException e) {
            return ResultModelFactory.FAIL500("广告主id输入错误, 请检查是否正确");
        } catch (BusinessException e) {
            return ResultModelFactory.FAIL500(e.getMessage());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return ResultModelFactory.FAIL500("服务器错误：" + e.getMessage());
        } finally {
            if (tryLock) {
                EXPORT_IMG_LOCK.unlock();
            }
        }
    }


    @GetMapping("/checkExport")
    public ResultModel<String> exportImgs1(@RequestParam(value = "advertiserIds", required = false) String advertiserIds) {
        try {
//            if (StringUtils.isBlank(advertiserIds)) {
//                return ResultModelFactory.FAIL400("暂不支持导出全部");
//            }
            List<Long> searchAdvertiserIds = Lists.newArrayList();
            if (StringUtils.isNotBlank(advertiserIds)) {
                if (StringUtils.contains(advertiserIds, " ")){
                    return ResultModelFactory.FAIL400("广告主id不能有空格！");
                }
                searchAdvertiserIds = Arrays.asList(advertiserIds.split(",")).stream().map(a -> Long.valueOf(a)).collect(Collectors.toList());
            }

            // 查出广告主营业执照
            List<AdvertiserDto> advertiserList = adShopIntegration.getAdShopList(new AdvertiserDto());

            Map<Integer,String> advertiserNameMap = Maps.newHashMap();
            advertiserNameMap = advertiserList.stream().collect(Collectors.toMap(AdvertiserDto::getId, a -> a.getCompany() == null ? "" : a.getCompany()));

            // 查出不存在的广告主id 后面需要返回
            StringBuilder notExistAdvertiser = new StringBuilder();
            if (CollectionUtils.isNotEmpty(searchAdvertiserIds)) {
                for (Long searchAdvertiserId : searchAdvertiserIds) {
                    if (!advertiserNameMap.containsKey(searchAdvertiserId.intValue())) {
                        notExistAdvertiser.append(searchAdvertiserId).append("  ");
                    }
                }
            }
            if (StringUtils.isNotBlank(notExistAdvertiser.toString())) {
                return ResultModelFactory.FAIL500("不存在的广告主id:" + notExistAdvertiser.toString());
            }
            return ResultModelFactory.SUCCESS();
        } catch (NumberFormatException e) {
            return ResultModelFactory.FAIL500("广告主id输入错误, 请检查是否正确");
        } catch (BusinessException e) {
            return ResultModelFactory.FAIL500(e.getMessage());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return ResultModelFactory.FAIL500("服务器错误：" + e.getMessage());
        }
    }


    @GetMapping(value = "/getAdvertiserQualifyInfo", produces = "application/json")
    public ResultModel getAdvertiserQualifyInfo(@RequestParam("advertiserId") Integer advertiserId) {
        AdvertiserAuditVo advertiserQualifyInfo = advertiserAuditService.getAdvertiserQualifyInfo(advertiserId);
        return ResultModelFactory.SUCCESS(advertiserQualifyInfo);
    }

    /**
     * 更新特批状态
     * @param advertiserId
     * @param specialApproval 是否特批 0.否 1.是
     * @return
     */
    @PostMapping(value = "/updateSpecialApproval", produces = "application/json")
    public ResultModel updateSpecialApproval(@RequestParam("advertiserId") Integer advertiserId, @RequestParam("specialApproval") Byte specialApproval) {
        if (!checkUpdateSpecialApproval(advertiserId, specialApproval)) {
            return ResultModelFactory.FAIL400("主体信息和资质信息均有一条审核通过才可打标");
        }
        return advertiserAuditService.updateSpecialApproval(advertiserId, specialApproval);
    }


    /**
     * 检查主体和资质都需要有审核通过
     * @param advertiserId
     * @param specialApproval
     * @return
     */
    boolean checkUpdateSpecialApproval(Integer advertiserId, Byte specialApproval) {
        if (Objects.equals(specialApproval, Constants.NO)) {
            return true;
        }
        AdvertiserMainInfoVo advertiserMainInfo = adShopIntegration.getAdvertiserMainInfo(advertiserId);
        if (Objects.equals(advertiserMainInfo.getStatus(), Constants.YES)) {
            List<TblAdQualify> qualifys = adQualifyMapperExt.getQualifyByAdvertiserIds(Lists.newArrayList(advertiserId.longValue()));
            for (TblAdQualify qualify : qualifys) {
                Integer auditStatus = qualify.getAuditStatus();
                if (Objects.equals(TblAdQualify.STATUS_AUDIT_PASS, auditStatus)) {
                    return true;
                }
            }
        }
        return false;
    }


}
