package com.bxm.pangu.rta.common.meituan;

import com.bxm.pangu.rta.common.*;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;

import java.util.List;

/**
 * @author allen
 * @date 2022-02-14
 * @since 1.0
 */
@Slf4j
public class MeituanRtaClient extends AbstractHttpClientRtaClient {

    public MeituanRtaClient(MeituanRtaProperties properties) {
        super(properties);
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        MeituanRtaProtos.RtaRequest.Builder builder = MeituanRtaProtos.RtaRequest.newBuilder();
        builder.setId(RandomStringUtils.randomAlphanumeric(8))
                .setIsPing(false)
                .setIsTest(false);

        MeituanRtaProtos.RtaRequest.Device.Builder b1 = MeituanRtaProtos.RtaRequest.Device.newBuilder();
        if (RtaRequest.Os.ANDROID.equalsIgnoreCase(request.getOs())) {
            b1.setOs(MeituanRtaProtos.RtaRequest.OperatingSystem.OS_ANDROID);
        } else if (RtaRequest.Os.IOS.equalsIgnoreCase(request.getOs())) {
            b1.setOs(MeituanRtaProtos.RtaRequest.OperatingSystem.OS_IOS);
        } else {
            b1.setOs(MeituanRtaProtos.RtaRequest.OperatingSystem.OS_UNKNOWN);
        }

        if (StringUtils.isNotBlank(request.getImei_md5())) {
            b1.setImeiMd5Sum(request.getImei_md5());
        }
        if (StringUtils.isNotBlank(request.getIdfa_md5())) {
            b1.setIdfaMd5Sum(request.getIdfa_md5());
        }
        if (StringUtils.isNotBlank(request.getAndroidid_md5())) {
            b1.setAndroidIdMd5Sum(request.getAndroidid_md5());
        }
        if (StringUtils.isNotBlank(request.getOaid_md5())) {
            b1.setOaidMd5Sum(request.getOaid_md5());
        }
        if (StringUtils.isNotBlank(request.getOaid())) {
            b1.setOaid(request.getOaid());
        }

        builder.setDevice(b1.build());
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("param");
        }
        String[] arr = StringUtils.split(param, "|");
        builder.setSiteId(arr[0]);

        MeituanRtaProtos.RtaRequest req = builder.build();

        HttpPost post = new HttpPost(getProperties().getUrl());

        post.addHeader("Content-Type", "application/x-protobuf;charset=UTF-8");
        post.setEntity(new ByteArrayEntity(req.toByteArray()));

        return post;
    }

    @Override
    protected boolean isTarget(RtaRequest request, byte[] bytes) {
        try {
            MeituanRtaProtos.RtaResponse response = MeituanRtaProtos.RtaResponse.parseFrom(bytes);
            int code = response.getCode();

            List<Long> targetIdList = response.getPromotionTargetIdList();
            if (log.isDebugEnabled()) {
                log.debug("target: {}", targetIdList);
            }
            String param = request.getParam();
            String[] arr = StringUtils.splitPreserveAllTokens(param, "|");
            if (arr.length < 2) {
                throw new RtaRequestException("param value length must >= 2");
            }

            return 0 == code && targetIdList.contains(NumberUtils.toLong(arr[1]));
        } catch (InvalidProtocolBufferException e) {
            log.warn("parseFrom: {}", e.getMessage());
        }
        return false;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        throw new UnsupportedOperationException();
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Meituan;
    }

    @Override
    public ProtocolFormat getProtocolFormat() {
        return ProtocolFormat.Protobuf;
    }
}
