package com.bxm.pangu.rta.common.oneway;

import com.bxm.pangu.rta.common.AbstractHttpClientRtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;

/**
 * OneWay rta 实现
 *
 * <p>Doc: http://doc.oneway.mobi/RTA/#/</p>
 *
 * @author allen
 * @date 2021-08-04
 * @since 1.1.4
 */
public class OneWayRtaClient extends AbstractHttpClientRtaClient {

    private final OneWayRtaProperties properties;

    public OneWayRtaClient(OneWayRtaProperties properties) {
        super(properties);
        this.properties = properties;
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.OneWay;
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String os = request.getOs();
        String platform = StringUtils.equals(os, RtaRequest.Os.ANDROID) ? "1"
                : StringUtils.equals(os, RtaRequest.Os.IOS) ? "2"
                : "0";

        String param = request.getParam();
        String[] arr = split(param);
        String[] ids = StringUtils.split(arr[1], ",");

        Map<String, Object> params = Maps.newHashMap();
        params.put("channel", arr[0]);
        params.put("offerIds", ids);
        params.put("imeiMd5", request.getImei_md5());
        params.put("oaid", request.getOaid());
        params.put("oaidMd5", request.getOaid_md5());
        params.put("idfa", request.getIdfa());
        params.put("idfaMd5", request.getIdfa_md5());
        params.put("platform", platform);

        HttpPost post = new HttpPost(properties.getUrl());
        String entity = JsonHelper.convert(params);
        post.setEntity(new StringEntity(entity, StandardCharsets.UTF_8));
        post.addHeader("Content-Type", "application/json;charset=UTF-8");
        post.addHeader("Accept-Encoding", "gzip");
        return post;
    }

    private String[] split(String param) {
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("param must be {channel}|{offerIds,offerIds}");
        }
        String[] arr = StringUtils.splitPreserveAllTokens(param, "|");
        if (arr.length < 2) {
            throw new RtaRequestException("param must be {channel}|{offerIds,offerIds}");
        }
        return arr;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        Matched matched = JsonHelper.convert(json, Matched.class);
        if (!matched.isSuccess()) {
            return false;
        }
        MatchedData data = matched.getData();
        if (Objects.isNull(data)) {
            return false;
        }
        List<String> offerIds = data.getOfferIds();
        if (CollectionUtils.isEmpty(offerIds)) {
            return false;
        }
        String[] arr = split(request.getParam());
        ArrayList<String> configOfferIds = Lists.newArrayList(StringUtils.split(arr[1], ","));
        return configOfferIds.stream().anyMatch(new Predicate<String>() {
            @Override
            public boolean test(String s) {
                return offerIds.contains(s);
            }
        });
    }

    @Data
    private static class Matched {

        private int code;
        private boolean success;
        private String message;
        private MatchedData data;
    }

    @Data
    private static class MatchedData {

        private List<String> offerIds;
    }
}
