package com.bxm.pangu.rta.common.taobao_activity;

import com.bxm.pangu.rta.common.*;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.SafeMapHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.taobao.api.DefaultTaobaoClient;
import com.taobao.api.TaobaoClient;
import com.taobao.api.internal.util.TaobaoLogger;
import com.taobao.api.request.TbkRtaConsumerMatchRequest;
import com.taobao.api.response.TbkRtaConsumerMatchResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2022-04-11
 * @since 1.0
 */
@Slf4j
public class TaobaoActivityRtaClient implements RtaClient {

    private final TaobaoActivityRtaProperties properties;
    private final ConcurrentHashMap<String, TaobaoClient> clients = new ConcurrentHashMap<>();

    public TaobaoActivityRtaClient(TaobaoActivityRtaProperties properties) {
        this.properties = properties;
        TaobaoLogger.setNeedEnableLogger(properties.isNeedEnableLogger());
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.TaobaoActivity;
    }

    @Override
    public RtaClientProperties getProperties() {
        return properties;
    }

    private TaobaoClient getClient(String appKey, String appSecret) {
        return SafeMapHelper.get(clients, appKey, (SafeMapHelper.InitializingValue<TaobaoClient>) () -> {
            String url = properties.getUrl();
            int connectTimeout = properties.getConnectTimeout();
            int socketTimeout = properties.getSocketTimeout();
            return new DefaultTaobaoClient(url, appKey, appSecret, "json", connectTimeout, socketTimeout);
        });
    }

    @Override
    public boolean isTarget(RtaRequest request, Consumer<Map<Object, Object>> feedback) throws RtaRequestException {
        String param = request.getParam();
        String[] arr = StringUtils.splitPreserveAllTokens(param, "|");
        if (ArrayUtils.isEmpty(arr) || arr.length < 2) {
            throw new RtaRequestException("param");
        }
        long adzoneId = NumberUtils.toLong(arr[0]);
        String offerId = arr[1];

        TbkRtaConsumerMatchRequest req = new TbkRtaConsumerMatchRequest();
        req.setAdzoneId(adzoneId);

        TbkRtaConsumerMatchRequest.OfferList offer = new TbkRtaConsumerMatchRequest.OfferList();
        offer.setOfferId(offerId);

        req.setOfferList(Lists.newArrayList(offer));

        String oaidMd5 = request.getOaid_md5();
        if (StringUtils.isBlank(req.getDeviceValue()) && StringUtils.isNotBlank(oaidMd5)) {
            req.setDeviceValue(oaidMd5);
            req.setDeviceType("OAID");
        }

        String imeiMd5 = request.getImei_md5();
        if (StringUtils.isBlank(req.getDeviceValue()) && StringUtils.isNotBlank(imeiMd5)) {
            req.setDeviceValue(imeiMd5);
            req.setDeviceType("IMEI");
        }

        String idfaMd5 = request.getIdfa_md5();
        if (StringUtils.isBlank(req.getDeviceValue()) && StringUtils.isNotBlank(idfaMd5)) {
            req.setDeviceValue(idfaMd5);
            req.setDeviceType("IDFA");
        }

        try {
            String appKey = properties.getAppKey();
            String appSecret = properties.getAppSecret();
            if (arr.length > 2) {
                appKey = arr[2];
            }
            if (arr.length > 3) {
                appSecret = arr[3];
            }

            TaobaoClient client = getClient(appKey, appSecret);
            TbkRtaConsumerMatchResponse rsp = client.execute(req);

            TbkRtaConsumerMatchResponse.Data data = rsp.getData();
            if (log.isDebugEnabled()) {
                log.debug("response: {}", JsonHelper.convert(rsp));
            }
            if (Objects.isNull(data)) {
                return false;
            }
            List<TbkRtaConsumerMatchResponse.Resultlist> resultList = data.getResultList();
            if (CollectionUtils.isEmpty(resultList)) {
                return false;
            }
            Optional<TbkRtaConsumerMatchResponse.Resultlist> first = resultList.stream().filter(resultlist -> "1".equals(resultlist.getStatus())).findFirst();
            if (!first.isPresent()) {
                return false;
            }
            TbkRtaConsumerMatchResponse.Resultlist result = first.get();

            String clickUrl = result.getClickUrl();

            if (Objects.nonNull(feedback)) {
                Map<Object, Object> map = Maps.newHashMap();
                map.put("clickUrl", clickUrl);
                feedback.accept(map);
            }

            return true;
        } catch (Exception e) {
            throw new RtaRequestException(e);
        }
    }
}
