package com.bxm.spider.deal.common.utils;

import com.bxm.spider.deal.common.constant.RulerConstant;
import com.bxm.spider.utils.DateUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.util.regex.Pattern.compile;


/**
 * <p> url处理相关接口 </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2018/10/18
 */
public class StringHelp {
    private static String[] SEPARATORS = {">>", ">", " "};
    private static String SEPARATOR = " ";

    /**
     * 获取表格后缀
     * @date: 2019/3/6
     * @param id
     * @return java.lang.String
     * @throws 
     */
    public static String getSuffixModTen(Long id){
        if(null == id){
            return "";
        }
        return String.valueOf(id % 10);
    }
    /**
     * 获取 url 的 hashcode值 除10求余
     *
     * @param url
     * @return String
     * @throws
     * @date: 2018/10/18
     */
    public static String getHashCodeModTen(String url) {
        if (StringUtils.isBlank(url)) {
            return "";
        }
        return String.valueOf(Math.abs(url.hashCode() % 10));
    }

    /**
     * url 参数清洗
     * 1：去除 # 后面的参数
     * 2：参数进行排序
     *
     * @param url
     * @return
     */
    public static String clearUrl(String url) {
        if (StringUtils.isBlank(url)) {
            return "";
        }
        url = StringEscapeUtils.unescapeHtml(url);

        url = StringEscapeUtils.unescapeJava(url);

        // 从 url 中获取参数
        String[] urlArray = url.split("[?]");
        if (urlArray.length < 2) {
            return url;
        }

        // # 后面无用参数去除
        String[] paraArray = urlArray[1].split("#");
        String[] parameters = paraArray[0].split("&");
        if (parameters.length == 1) {
            return urlArray[0] + "?" + parameters[0];
        }

        // 参数排序
        Arrays.sort(parameters);
        StringBuffer paraSb = new StringBuffer();
        for (String parameter : parameters) {
            paraSb.append(parameter).append("&");
        }
        if (paraSb.length() > 0) {
            paraSb.setLength(paraSb.length() - 1);
        }

        return urlArray[0] + "?" + paraSb.toString();
    }

    /**
     * 把字符串的连接符缓存一个空格
     *
     * @param str
     * @return java.lang.String
     * @throws
     * @date: 2018/10/26
     */
    public static String splitSpace(String str) {
        if (StringUtils.isBlank(str)) {
            return null;
        }
        String[] strArray = {};
        for (String separator : SEPARATORS) {
            strArray = str.split(separator);
            if (strArray.length > 1) {
                break;
            }
        }

        StringBuffer sb = new StringBuffer();
        for (String s : strArray) {
            if (StringUtils.isNotBlank(s)) {
                sb.append(s.trim()).append(SEPARATOR);
            }
        }
        if (sb.length() > 0) {
            sb.setLength(sb.length() - 1);
        }
        return sb.toString();

    }

    /**
     * 从 url 中获取 page 参数并加 1 返回
     *
     * @param url  请求链接
     * @param page 分页参数
     * @return java.lang.String
     * @throws
     * @date: 2018/10/29
     */
    public static String analysisNextPage(String url, String page) {
        if (StringUtils.isBlank(page)) {
            return null;
        }
        // 从 url 中获取参数
        String[] urlArray = url.split("[?]");
        if (urlArray.length < 2) {
            return url;
        }

        // 分页参数加1处理
        String[] parameters = urlArray[1].split("&");
        for (String parameter : parameters) {
            String[] param = parameter.split("=");
            if (page.equals(param[0].trim())) {
                int current = Integer.parseInt(param[1].trim());
                return url.replace(parameter, page + "=" + (++current));
            }
        }
        return null;
    }

    public static void main(String[] args) {
        System.out.println(StringHelp.convertTimestampByBefore("4分前"));
    }
    /**
     * 指定长度 获取 UUID hashCode 值
     * 最大长度 20
     * @param length
     * @return java.lang.Long
     * @throws
     * @date: 2018/10/29
     */
    public static Long getUUIDHashCode(int length) {
        if(length >= 20){
            return null;
        }
        int randNo = UUID.randomUUID().toString().hashCode();
        if (randNo < 0) {
            randNo = -randNo;
        }
        if(String.valueOf(randNo).length() > length){
            randNo =  getLengthNum(randNo,length);
        }
        if(String.valueOf(randNo).length() == length){
            return Long.valueOf(randNo);
        }
        return Long.valueOf(randNo + String.format("%1$0" + (length-String.valueOf(randNo).length()) + "d", 0));
    }
    /**
     * 让数字的位数降低至指定长度
     * @date: 2019/3/7
     * @param source
     * @param length
     * @return int
     * @throws 
     */
    private static int getLengthNum(int source,int length){
        if(String.valueOf(source).length() <= length){
            return source;
        }
        int result = source >> 2;
        if(String.valueOf(result).length() > length){
            return getLengthNum(result,length);
        }
        return result;

    }

    /**
     * 根据视频id，获取解密后的视频 id
     *
     * @param video_id
     * @param secret
     * @return java.lang.String
     * @throws
     * @date: 2018/10/31
     */
    public static String formatVideoId(String video_id, String secret) {
        // 密钥分割
        List<Integer> keyList = new ArrayList<Integer>();
        String[] keyArray = secret.split("");
        for (String key : keyArray) {
            keyList.add((int) key.charAt(0));
        }
        int len = keyList.size();
        // 视频id分割
        String[] idArray = video_id.split(":");
        List<Integer> idList = new ArrayList<Integer>();
        for (int i = 1; i < idArray.length; i++) {
            idList.add(Integer.parseInt(idArray[i]) - (255 & keyList.get((i - 1) % len)));
        }
        // 新的视频 id 组合
        StringBuffer result = new StringBuffer();
        for (Integer id : idList) {
            result.append((char) id.intValue());
        }
        return result.toString();
    }

    /**
     * 把年份转换成系统当年
     *
     * @param date
     * @return java.lang.String
     * @throws
     * @date: 2018/11/10
     */
    public static String replaceYear(String date) {
        if (StringUtils.isBlank(date)) {
            return "";
        }
        if (date.length() < 4) {
            return date;
        }
        Calendar calendar = Calendar.getInstance();
        return date.replaceAll("\\d{4}", String.valueOf(calendar.get(Calendar.YEAR)));
    }
    /**
     * 把月份转换成系统当月
     * @date: 2018/12/10
     * @param date
     * @return java.lang.String
     * @throws
     */
    public static String replaceMonth(String date) {
        if (StringUtils.isBlank(date)) {
            return "";
        }
        if (date.length() < 7) {
            return date;
        }
        Calendar calendar = Calendar.getInstance();
        String month = String.valueOf(calendar.get(Calendar.MONTH) + 1).length() < 2 ? "0" + String.valueOf(calendar.get(Calendar.MONTH) + 1) : String.valueOf(calendar.get(Calendar.MONTH) + 1);
        return date.replaceAll("(-)(\\d{2})(-)", "$1"+month+"$3");
    }
    /**
     * 把日转换成系统当日
     * @date: 2018/12/10
     * @param date
     * @return java.lang.String
     * @throws
     */
    public static String replaceDay(String date) {
        if (StringUtils.isBlank(date)) {
            return "";
        }
        if (date.length() < 11) {
            return date;
        }
        Calendar calendar = Calendar.getInstance();
        String day = String.valueOf(calendar.get(Calendar.DATE)).length() < 2 ? "0" + String.valueOf(calendar.get(Calendar.DATE)) : String.valueOf(calendar.get(Calendar.DATE));
        return date.replaceAll("(-)(\\d{2})(\\s)", "$1"+day+"$3");
    }

    /**
     * @param date      把时间按新的格式转换
     * @param patterOld
     * @param patterNew
     * @return java.lang.String
     * @throws
     * @date: 2018/11/10
     */
    public static String convertDateStrToStr(String date, String patterOld, String patterNew) {
        if (StringUtils.isBlank(date) || StringUtils.isBlank(date) || StringUtils.isBlank(date)) {
            return date;
        }
        Date oldDate = DateUtils.convertStringToDate(date.trim(), patterOld);
        String newDate = DateUtils.convertDateToString(oldDate, patterNew);
        if (patterOld.indexOf("yyyy") == -1) {
            newDate = replaceYear(newDate);
        }
        if (patterOld.indexOf("MM") == -1) {
            newDate = replaceMonth(newDate);
        }
        if (patterOld.indexOf("dd") == -1) {
            newDate = replaceDay(newDate);
        }

        // 转换后的时间大于当前时间，需要对年进行-1（无年有月日的格式转换后问题处理）
        boolean isNewDateOverNow = DateUtils.convertStringToDate(newDate, patterNew).getTime() > System.currentTimeMillis();
        if (isNewDateOverNow && patterOld.indexOf("yyyy") == -1) {
            newDate = reduceOneYear(patterNew, newDate);
        }
        return newDate;
    }

    /**
     * 减一年
     * @param patterNew
     * @param newDate
     * @return
     */
    private static String reduceOneYear(String patterNew, String newDate) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(DateUtils.convertStringToDate(newDate,patterNew));
        calendar.add(Calendar.YEAR,-1);
        newDate = DateUtils.convertDateToString(calendar.getTime(),patterNew);
        return newDate;
    }

    /**
     * 将时间戳转为新的格式
     * @param timestamp
     * @return
     */
    public static String convertTimestampToStr(long timestamp) {
        Date date = new Date(timestamp);
        return DateUtils.convertDateToString(date, RulerConstant.TIME_PATTER);
    }
    /**
     * 将之前的时间转换成新的格式
     * @date: 2019/1/16
     * @param content
     * @return java.lang.String
     * @throws 
     */
    public static String convertTimestampByBefore(String content) {
        if(StringUtils.isBlank(content)){
            return "";
        }
        Date date = null;
        if(content.indexOf("前天") != -1){
            date = DateUtils.setField(new Date(),Calendar.DATE,-2);
        }
        if(content.indexOf("昨天") != -1){
            date = DateUtils.setField(new Date(),Calendar.DATE,-1);
        }
        if(content.indexOf("今天") != -1){
            date = new Date();
        }
        if(null == date && content.indexOf("前") != -1){
            if(content.indexOf("天") != -1){
                String[] timestamp = content.trim().split("天");
                if(StringUtils.isNumeric(null==timestamp[0]?null:timestamp[0].trim())){
                    date = DateUtils.setField(new Date(),Calendar.DAY_OF_YEAR,Integer.valueOf(timestamp[0].trim())*(-1));
                }
            }
            if(content.indexOf("小时") != -1){
                String[] timestamp = content.trim().split("小");
                if(null != timestamp[0] && timestamp[0].indexOf(".") != -1){
                    timestamp = timestamp[0].split("\\.");
                }
                if(StringUtils.isNumeric(null==timestamp[0]?null:timestamp[0].trim())){
                    date = DateUtils.setField(new Date(),Calendar.HOUR_OF_DAY,Integer.valueOf(timestamp[0].trim())*(-1));
                }
            }
            if(content.indexOf("分") != -1){
                String[] timestamp = content.trim().split("分");
                if(StringUtils.isNumeric(null==timestamp[0]?null:timestamp[0].trim())){
                    date = DateUtils.setField(new Date(),Calendar.MINUTE,Integer.valueOf(timestamp[0].trim())*(-1));
                }
            }
        }
        if(null != date){
            return DateUtils.formatAtWill(date,DateUtils.DAY_HOUR_MINUTE_SECOND);
        }
        return "";
    }

    public static String trim(String str){
        if(null == str){
            return "";
        }
        return str.trim();
    }

    public static boolean isDigits(String str){
        String regex = "\\d+(\\.\\d+)?";
        return str.matches(regex);
    }

    /**
     * 判断是否包含中文
     *
     * @param str
     * @return boolean
     * @throws
     * @date: 2018/11/28
     */
    public static boolean isContainChinese(String str) {
        if(StringUtils.isBlank(str)){
            return false;
        }
        Pattern p = compile("[\u4e00-\u9fa5]");
        Matcher m = p.matcher(str);
        if (m.find()) {
            return true;
        }
        return false;
    }

    /**
     * 判断是否包含英文字母
     *
     * @param str
     * @return boolean
     * @throws
     * @date: 2018/11/28
     */
    public static boolean isContainLetter(String str) {
        if(StringUtils.isBlank(str)){
            return false;
        }
        Pattern p = compile("[a-zA-z]");
        Matcher m = p.matcher(str);
        if (m.find()) {
            return true;
        }
        return false;
    }
    /**
     * 判断是否包含数字
     * @date: 2018/12/6
     * @param str
     * @return boolean
     * @throws
     */
    public static boolean isContainNumber(String str) {
        if(StringUtils.isBlank(str)){
            return false;
        }
        Pattern p = compile("[0-9]");
        Matcher m = p.matcher(str);
        if (m.find()) {
            return true;
        }
        return false;
    }
}
