package com.bxm.spider.deal.timer.job.jobs;

import com.alibaba.fastjson.JSONObject;
import com.bxm.spider.deal.common.utils.StringHelp;
import com.bxm.spider.deal.dal.service.CompanyService;
import com.bxm.spider.deal.dal.service.WorkService;
import com.bxm.spider.deal.model.dao.Company;
import com.bxm.spider.deal.model.dao.Work;
import com.bxm.spider.deal.model.vo.RecruitMessageVo;
import com.bxm.spider.deal.model.vo.WorkVo;
import com.bxm.spider.deal.timer.job.Job;
import com.bxm.spider.deal.timer.job.config.RecruitJobConfiguration;
import com.bxm.spider.deal.timer.job.constant.RecruitConstant;
import com.bxm.spider.deal.timer.job.utils.HttpUtil;
import com.bxm.spider.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.Scheduled;

import java.math.BigDecimal;
import java.text.ParseException;
import java.util.*;

/**
 * <p> 招聘信息同步job </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2018/11/21
 */
@Configuration
@EnableConfigurationProperties(RecruitJobConfiguration.class)
public class RecruitMessageJob implements Job{
    private Logger logger = LoggerFactory.getLogger(RecruitMessageJob.class);
    @Autowired
    private WorkService workService;
    @Autowired
    private CompanyService companyService;
    @Autowired
    private RecruitJobConfiguration recruitConfig;

    private static final String POST_PARA_NAME = "crawlerInfo";
    private Map<Long,String> map = null;//搜集本次同步不符合要求的数据
    private int count = 0;//统计：工资面议的数量

    @Scheduled(cron = "${scheduling.recruit.time}")
    @Override
    public void execute() {
        Long start = System.currentTimeMillis();
        //获取指定时间内更新的数据
        List<Work> workList = workService.selectByModifyTime(recruitConfig.getTimeBefore());
        if(null == workList || workList.size() == 0){
            logger.info("【同步招聘信息】无数据更新，不同步");
            return;
        }

        map = new HashMap<>();
        count = 0;

        //获取公司url
        List<String> companyUrls = new ArrayList<>();
        Map<String,List<WorkVo>> workVoMap = new HashMap<>();
        for(Work work : workList){
            if(null == work){
                continue;
            }
            String msg = checkMsg(work);
            if(StringUtils.isNotBlank(msg)){
                logger.warn("【同步招聘信息】信息不符合要求，信息为：{}，work:{}",msg,work);
                count++;
                continue;
            }
            WorkVo workVo = copyWorkMsg(work);
            if(null == workVo){
                logger.warn("【同步招聘信息】工作信息copy失败，work参数：{}",work);
                continue;
            }
            companyUrls.add(work.getCompanyUrl());

            if(workVoMap.containsKey(work.getCompanyUrl())){
                workVoMap.get(work.getCompanyUrl()).add(workVo);
            } else {
                List<WorkVo> workVoList= new ArrayList<>();
                workVoList.add(workVo);
                workVoMap.put(work.getCompanyUrl(),workVoList);
            }
        }

        if(null == companyUrls || companyUrls.size() == 0){
            logger.warn("【同步招聘信息】公司信息为空，耗时：{} 毫秒",(System.currentTimeMillis() - start));
            count = 0;
            map = null;
            return;
        }

        //获取公司信息  并添加对应的招聘信息
        List<Company> companyList = companyService.selectByUrl(companyUrls);
        List<RecruitMessageVo> messageVoList = new ArrayList<>();
        for(Company company : companyList){
            if(null == company || StringUtils.isBlank(company.getAddress())){
                logger.warn("【同步招聘信息】公司为null 或地址为空：{}",company);
            }
            if(!workVoMap.containsKey(company.getUrl())){
                continue;
            }
            if(null == workVoMap.get(company.getUrl())){
                continue;
            }
            RecruitMessageVo messageVo = copyCompanyMsg(company,workVoMap.get(company.getUrl()));
            if(null == messageVo){
                logger.warn("【同步招聘信息】公司信息全：{}",company);
                continue;
            }
            messageVoList.add(messageVo);
            logger.info("【同步招聘信息】公司：{} 对应的岗位数为：{}",company.getCompanyName(),workVoMap.get(company.getUrl()).size());
        }

        if(null != messageVoList && messageVoList.size() > 0){
            Map<String,Object> para = new HashMap<>();
            para.put(POST_PARA_NAME,JSONObject.toJSONString(messageVoList));
            if(logger.isDebugEnabled()){
                logger.debug("【同步招聘信息】参数为：{},url:{}",JSONObject.toJSONString(para),recruitConfig.getUrl());
            }
            String result = HttpUtil.okHttpPost(recruitConfig.getUrl(), para);
            if(StringUtils.isNotBlank(result)){
                JSONObject jsonObject = JSONObject.parseObject(result);
                if(null != jsonObject && null != jsonObject.get("returnValue") && (Boolean) jsonObject.get("returnValue")){
                    logger.info("【同步招聘信息】同步公司和岗位信息成功，返回结果为：{},同步公司数为：{}，dimCount:{},map:{},耗时：{} 毫秒", result, messageVoList.size(),count,map, (System.currentTimeMillis() - start));
                } else {
                    logger.error("【同步招聘信息】同步公司和岗位信息失败，返回结果:{},同步公司数为：{}，耗时：{} 毫秒", result,messageVoList.size(), (System.currentTimeMillis() - start));
                }
            } else {
                logger.error("【同步招聘信息】同步公司和岗位信息结束，返回结果为null,同步公司数为：{}，耗时：{} 毫秒", messageVoList.size(), (System.currentTimeMillis() - start));
            }
        } else {
            logger.warn("【同步招聘信息】公司信息获取为空，无需同步，耗时：{} 毫秒",(System.currentTimeMillis() - start));
        }

        count = 0;
        map = null;
    }

    /**
     * 获取同步的公司信息
     * @date: 2018/11/21
     * @param company
     * @param workList
     * @return com.bxm.spider.deal.model.vo.RecruitMessageVo
     * @throws 
     */
    private RecruitMessageVo copyCompanyMsg(Company company,List<WorkVo> workList) {
        if(null == workList){
            logger.error("【同步招聘信息】公司对应的岗位信息为null");
            return null;
        }
        boolean flag = false;
        if(StringUtils.isBlank(company.getAddress())){
            flag = true;
        }
        RecruitMessageVo  messageVo = new RecruitMessageVo();
        for(int i=0; i<workList.size(); i++){
            WorkVo workVo = workList.get(i);
            if(null == workVo){
                continue;
            }
            if(flag && StringUtils.isNotBlank(workVo.getAddress())){
                company.setAddress(workVo.getAddress());
                flag = false;
            }
            if(StringUtils.isNotBlank(workVo.getMobile())){
                continue;
            }
            if(StringUtils.isBlank(company.getMobile()) && StringUtils.isBlank(company.getPhone())){
                workList.remove(i);
                logger.warn("【同步招聘信息】手机号和电话为空：{}",workVo);
                map.put(workVo.getCrawlerId(),company.getCompanyName());
                continue;
            }
            String mobile = getRightMobile(company.getMobile(),company.getPhone());
            if(StringUtils.isBlank(mobile)){
                workList.remove(i);
                logger.warn("【同步招聘信息】手机号和电话为格式有误：{}",workVo);
                map.put(workVo.getCrawlerId(),company.getCompanyName());
                continue;
            }
            workVo.setMobile(mobile);
        }
        if(flag) {
            logger.warn("【同步招聘信息】公司地址为空");
            return null;
        }
        messageVo.setCrawlerId(company.getId());
        messageVo.setFullName(company.getCompanyName());
        messageVo.setShortName(company.getCompanyName());
        messageVo.setIndustry(company.getIndustry());
        messageVo.setScale(StringHelp.isContainChinese(company.getCompanyScale()) ?
                company.getCompanyScale() : company.getCompanyScale() + RecruitConstant.COMPANY_SCALE);
        messageVo.setNature(company.getCompanyType());
        if(StringUtils.isNotBlank(company.getLatitude()) && StringUtils.isNotBlank(company.getLongitude())) {
            try {
                messageVo.setAddressLat(new BigDecimal(company.getLatitude()));
                messageVo.setAddressLon(new BigDecimal(company.getLongitude()));
            } catch (Exception e) {
                logger.warn("【同步招聘信息】公司经纬度信息转换失败，经度：{},纬度：{}", company.getLongitude(), company.getLatitude(), e);
            }
        }
        messageVo.setAddress(company.getAddress());
        setArea(messageVo,company.getRegion());
        messageVo.setIntroduction(company.getDesc());
        messageVo.setLogo(company.getLogoUrl());
//        messageVo.setPic();
        messageVo.setState(company.getStatus());
        messageVo.setJobListInfo(workList);
        return messageVo;
    }

    /**
     * 区域处理
     * @date: 2018/11/29
     * @param messageVo
     * @param region
     * @return void
     * @throws 
     */
    private void setArea(RecruitMessageVo messageVo, String region) {
        if(StringUtils.isNotBlank(region)){
            for (String key : RecruitConstant.REGION_MAP.keySet()){
                if(region.indexOf(key) != -1){
                    region = RecruitConstant.REGION_MAP.get(key);
                    break;
                }
            }
            messageVo.setArea(region);
        }
    }

    /**
     * 同步需要的岗位信息
     * @date: 2018/11/21
     * @param work
     * @return com.bxm.spider.deal.model.vo.WorkVo
     * @throws 
     */
    private WorkVo copyWorkMsg(Work work) {
        WorkVo workVo = new WorkVo();
        workVo.setCrawlerId(work.getId());
        workVo.setWorkType(work.getWorkType());
        // 工资
        setSalary(workVo,work.getSalary());
        // 性别
        workVo.setGenderReq(getGenderFlag(work.getSex()));
        // 年龄
        setAge(workVo,work.getAge());
        workVo.setEduReq(work.getEducation());
//        workVo.setOtherReq();
        workVo.setWorkContent(work.getDesc());
//        workVo.setWorkHour();
        // 设置招聘人数
        setPeopleNum(workVo,work.getPeopleNum());
        workVo.setStartTime(work.getModifyTime());
        setEndTime(workVo,work.getModifyTime());
        // 手机号处理
        workVo.setMobile(getRightMobile(work.getMobile(),work.getPhone()));
        workVo.setState(work.getStatus());
        workVo.setAddress(work.getAddress());
        workVo.setHireMode(RecruitConstant.HIRE_MODEL_DEFAULT);
        return workVo;
    }

    /**
     * 手机号处理
     * @date: 2018/11/28
     * @param phones
     * @return java.lang.String
     * @throws 
     */
    private String getRightMobile(String ... phones) {
        for(String phone : phones){
            if(StringUtils.isBlank(phone)){
                continue;
            }
            if(phone.length() > 13 || phone.length() < 5){
                continue;
            }
            if(StringHelp.isContainChinese(phone)){
                continue;
            }
            if(StringHelp.isContainLetter(phone)){
                continue;
            }
            return phone;
        }
        return "";
    }

    private void setEndTime(WorkVo workVo, Date modifyTime) {
        try {
            if(null == modifyTime){
                modifyTime = new Date();
            }
            String end = DateUtils.getDate(DateUtils.convertDateToString(modifyTime, DateUtils.PATTERN_STR19),RecruitConstant.END_TIME_DEFAULT, DateUtils.PATTERN_STR19);
            workVo.setEndTime(DateUtils.convertStringToDate(end, DateUtils.PATTERN_STR19));
        } catch (ParseException e) {
            logger.warn("【同步招聘岗位信息】结束时间设置失败，modifyTime：{}", modifyTime, e);
        }
    }

    /**
     * 招聘人数转换
     * @param workVo
     * @param peopleNum
     */
    private void setPeopleNum(WorkVo workVo, String peopleNum) {
        try {
            workVo.setHireNum(StringUtils.isNotBlank(peopleNum) && StringUtils.isNumeric(peopleNum.replaceAll(RecruitConstant.PEOPLE_NUM, "").trim()) ?
                    Integer.valueOf(peopleNum.replaceAll(RecruitConstant.PEOPLE_NUM, "").trim()) : RecruitConstant.PEOPLE_NUM_DEFAULT);
        } catch (Exception e) {
            logger.warn("【同步招聘信息】招聘人数信息转换失败，error:",e);
            workVo.setHireNum(RecruitConstant.PEOPLE_NUM_DEFAULT);
        }
    }

    /**
     * 年龄分割
     * @date: 2018/11/22
     * @param workVo
     * @param age
     * @return void
     * @throws 
     */
    private void setAge(WorkVo workVo, String age) {
        String[] ageArray = age.split(RecruitConstant.AGE_SPLIT1);
        if(ageArray.length == 1){
            ageArray = age.split(RecruitConstant.AGE_SPLIT2);
        }
        if(ageArray.length == 2){
            try {
                workVo.setAgeMinReq(Integer.valueOf(ageArray[0].trim()));
                workVo.setAgeMaxReq(Integer.valueOf(ageArray[1].trim()));
                return;
            } catch (Exception e) {
                logger.error("【同步招聘信息】年龄要求信息转换失败，error:",e);
            }
        }
        workVo.setAgeMinReq(RecruitConstant.AGE_MIN);
        workVo.setAgeMaxReq(RecruitConstant.AGE_MAX);
    }

    /**
     * 工资分割
     * @date: 2018/11/22
     * @param workVo
     * @param salary
     * @return void
     * @throws 
     */
    private void setSalary(WorkVo workVo, String salary) {
        if(StringUtils.isNotBlank(salary)){
            //以上处理
            if(salary.indexOf(RecruitConstant.SALARY_YISHANG) != -1){
                for (String key : RecruitConstant.SALARY_MAP.keySet()){
                    if(salary.indexOf(key) != -1){
                        salary = salary.replaceAll(key,RecruitConstant.SALARY_MAP.get(key));
                    }
                }
            }
            //单位字符串处理
            for(String place : RecruitConstant.SALARY_PLACE_BEFORE){
                salary = salary.replaceAll(place,"");
            }
            //区间最大值最小值分割
            for (String spl : RecruitConstant.SALARY_SPLIT){
                String[] salArray = salary.split(spl);
                if(salArray.length == 2){
                    try {
                        workVo.setSalaryMin(Integer.valueOf(salArray[0].trim()));
                        workVo.setSalaryMax(Integer.valueOf(salArray[1].trim()));
                        return;
                    } catch (Exception e) {
                        logger.warn("【同步招聘信息】工资最大值最小值信息转换失败，salary:{}",salary,e);
                    }
                }
            }
            try {
                salary = salary.replaceAll(RecruitConstant.SALARY_PLACE_AFTER, "").trim();
                if(!StringUtils.isNumeric(salary)){
                    logger.warn("【同步招聘信息】工资格式有误，salary:{}，workVo:{}",salary,workVo);
                    return;
                }
                workVo.setSalaryMin(Integer.valueOf(salary.replaceAll(RecruitConstant.SALARY_PLACE_AFTER, "").trim()));
                workVo.setSalaryMax(Integer.valueOf(salary.replaceAll(RecruitConstant.SALARY_PLACE_AFTER, "").trim()) + RecruitConstant.SALARY_MAX);
            } catch (Exception e) {
                logger.error("【同步招聘信息】工资要求信息转换失败，salary:{},error:",salary,e);
            }

        }
    }

    /**
     * 获取性别标识
     * @date: 2018/11/21
     * @param sex
     * @return short
     * @throws 
     */
    private short getGenderFlag(String sex) {
        if(sex.indexOf(RecruitConstant.SEX_MAN) != -1 && sex.indexOf(RecruitConstant.SEX_WOMAN) == -1){
            return 1;
        }
        if(sex.indexOf(RecruitConstant.SEX_MAN) == -1 && sex.indexOf(RecruitConstant.SEX_WOMAN) != -1){
            return 2;
        }
        return 0;
    }

    private String checkMsg(Work work) {
        StringBuffer sb = new StringBuffer();
        if(null == work.getSalary() || !StringHelp.isContainNumber(work.getSalary())){
            sb.append("工资格式有误").append(",");
        }
        if(StringUtils.isBlank(work.getCompanyUrl())){
            sb.append("公司url为空").append(",");
        }
        if(sb.length() > 0){
            sb.setLength(sb.length() - 1);
        }
        return sb.toString();
    }
}
