package com.bxm.spider.prod.listener;

import org.quartz.JobExecutionContext;
import org.quartz.Trigger;
import org.quartz.TriggerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

/**
 * trigger listener
 *
 * @ClassName SpiderTriggerListener
 * @CopyRright (c) 2018-bxm：杭州微财网络科技有限公司
 * @Author kk.xie
 * @Date 2018/10/23 10:52
 * @Version 1.0
 * @Modifier kk.xie
 * @Modify Date 2018/10/23 10:52
 **/
@Component
public class SpiderTriggerListener implements TriggerListener {

    private Logger logger = LoggerFactory.getLogger(SpiderTriggerListener.class);

    /**
     * 返回一个字符串用以说明监听器的名称
     *
     * @param
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/23 11:29
     */
    @Override
    public String getName() {
        return SpiderTriggerListener.class.getName();
    }

    /**
     * (1) Trigger被激发 它关联的job即将被运行
     *
     * 当与监听器相关联的 Trigger 被触发，Job 上的 execute() 方法将要被执行时，Scheduler 就调用这个方法。
     * 在全局 TriggerListener 情况下，这个方法为所有 Trigger 被调用。
     *
     * @param trigger
     * @param context
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/23 11:29
     */
    @Override
    public void triggerFired(Trigger trigger, JobExecutionContext context) {
        logger.info("execute triggerFired" + trigger.toString());
    }

    /**
     * (2) Trigger被激发 它关联的job即将被运行,先执行(1)，在执行(2) 如果返回TRUE 那么任务job会被终止
     *
     * 在 Trigger 触发后，Job 将要被执行时由 Scheduler 调用这个方法。
     * TriggerListener 给了一个选择去否决 Job 的执行。假如这个方法返回 true，这个 Job 将不会为此次 Trigger 触发而得到执行
     *
     * @param trigger
     * @param context
     * @return boolean
     * @throws
     * @author kk.xie
     * @date 2018/10/23 11:30
     */
    @Override
    public boolean vetoJobExecution(Trigger trigger, JobExecutionContext context) {
        logger.info("execute vetoJobExecution" + trigger.toString());
        return false;
    }

    /**
     * (3) 当Trigger错过被激发时执行,比如当前时间有很多触发器都需要执行，但是线程池中的有效线程都在工作，
     *  那么有的触发器就有可能超时，错过这一轮的触发。
     *
     * Scheduler 调用这个方法是在 Trigger 错过触发时。
     * 如这个方法的 JavaDoc 所指出的，你应该关注此方法中持续时间长的逻辑：在出现许多错过触发的 Trigger 时，长逻辑会导致骨牌效应。你应当保持这上方法尽量的小
     *
     * @param trigger
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/23 11:31
     */
    @Override
    public void triggerMisfired(Trigger trigger) {
        logger.info("execute triggerMisfired" + trigger.toString());
    }

    /**
     * (4) 任务完成时触发
     *
     * Trigger 被触发并且完成了 Job 的执行时，Scheduler 调用这个方法。
     * 这不是说这个 Trigger 将不再触发了，而仅仅是当前 Trigger 的触发(并且紧接着的 Job 执行) 结束时。这个 Trigger 也许还要在将来触发多次的
     *
     * @param trigger
     * @param context
     * @param triggerInstructionCode
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/10/23 11:32
     */
    @Override
    public void triggerComplete(Trigger trigger, JobExecutionContext context, Trigger.CompletedExecutionInstruction triggerInstructionCode) {
        logger.info("execute triggerComplete" + trigger.toString());
    }
}
