package com.bxm.spider.prod.service.impl;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.bxm.spider.cache.RedisClient;
import com.bxm.spider.cache.constant.TaskKeyConstant;
import com.bxm.spider.constant.task.TaskStatusEnum;
import com.bxm.spider.prod.common.constants.ConfigStatusEnum;
import com.bxm.spider.prod.common.constants.Constant;
import com.bxm.spider.prod.job.SpiderDetailsJob;
import com.bxm.spider.prod.job.SpiderJob;
import com.bxm.spider.prod.job.SpiderQueueJob;
import com.bxm.spider.prod.model.dao.LoginAccount;
import com.bxm.spider.prod.model.dao.UrlConfig;
import com.bxm.spider.prod.model.dao.UrlTask;
import com.bxm.spider.prod.service.JobService;
import com.bxm.spider.prod.service.LoginAccountService;
import com.bxm.spider.prod.service.UrlConfigService;
import com.bxm.spider.prod.service.UrlTaskService;
import com.bxm.spider.prod.utils.ProdServiceUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.quartz.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * 任务维护实现
 *
 * @ClassName JobServiceImpl
 * @CopyRright (c) 2018-bxm：杭州微财网络科技有限公司
 * @Author kk.xie
 * @Date 2018/10/18 16:45
 * @Version 1.0
 * @Modifier kk.xie
 * @Modify Date 2018/10/18 16:45
 **/
@Service
public class JobServiceImpl implements JobService {

    private Logger logger = LoggerFactory.getLogger(JobServiceImpl.class);

    @Autowired
    private Scheduler scheduler;

    @Autowired
    private UrlTaskService urlTaskService;

    @Autowired
    private UrlConfigService urlConfigService;

    @Autowired
    private RedisClient redisClient;

    @Autowired
    private LoginAccountService loginAccountService;

    @Override
    public boolean addCronJob(String serialNum) {
        return addCronJob(serialNum, SpiderJob.class, SpiderQueueJob.class, SpiderDetailsJob.class);
    }

    @Override
    public boolean addCronJob(String serialNum, Class <? extends Job> jobClass, Class <? extends Job> jobQueueClass, Class <? extends Job> jobDetailsClass) {
        try {
            String jobGroup = getGroupBySerialNum(serialNum);
            String jobName = ProdServiceUtils.getJobName(serialNum);
            JobKey jobKey = JobKey.jobKey(jobName, jobGroup);

            JobDetail jobDetail = scheduler.getJobDetail(jobKey);

            if (jobDetail != null) {
                logger.warn("job: {} 已存在！", jobName);
                return false;
            }
            UrlTask urlTask = urlTaskService.selectOne(new EntityWrapper().eq("serial_num", serialNum));
            // 新建爬取周期任务
            jobDetail = initJobDetail(jobClass, jobName, jobGroup, serialNum);
            CronTrigger cronTrigger = initCronTrigger(jobName, jobGroup, urlTask.getSchule());
            scheduler.scheduleJob(jobDetail, cronTrigger);
            logger.info("add cron job: {} success！", jobName);

            // 新建列表爬取速率控制任务
            String queueJobName = ProdServiceUtils.getQueueJobName(serialNum);
            jobDetail = initJobDetail(jobQueueClass, queueJobName, jobGroup, serialNum);
            jobDetail.getJobDataMap().put(Constant.EXECUTE_RATE, urlTask.getQueueRate());
            cronTrigger = initCronTrigger(queueJobName, jobGroup, urlTask.getQueueSchule());
            scheduler.scheduleJob(jobDetail, cronTrigger);
            logger.info("add cron job: {} success！", queueJobName);

            // 新建详情爬取速率控制任务
            String detailsJobName = ProdServiceUtils.getDetailsJobName(serialNum);
            jobDetail = initJobDetail(jobDetailsClass, detailsJobName, jobGroup, serialNum);
            jobDetail.getJobDataMap().put(Constant.EXECUTE_RATE, urlTask.getDetailsRate());
            cronTrigger = initCronTrigger(detailsJobName, jobGroup, urlTask.getDetailsSchule());
            scheduler.scheduleJob(jobDetail, cronTrigger);
            logger.info("add cron job: {} success！", detailsJobName);
            return true;
        } catch (Exception e) {
            // 新增job异常则删除
            deleteJob(serialNum);
            logger.error("add cron job: {} error", ProdServiceUtils.getJobName(serialNum), e);
            return false;
        }
    }

    @Override
    public boolean pauseJob(String serialNum) {
        try {
            String jobGroup = getGroupBySerialNum(serialNum);
            String jobName = ProdServiceUtils.getJobName(serialNum);

            TriggerKey triggerKey = TriggerKey.triggerKey(jobName, jobGroup);
            scheduler.pauseTrigger(triggerKey);

            // 列表子任务一起暂停
            String queueJobName = ProdServiceUtils.getQueueJobName(serialNum);
            triggerKey = TriggerKey.triggerKey(queueJobName, jobGroup);
            scheduler.pauseTrigger(triggerKey);

            // 详情子任务一起暂停
            String detailsJobName = ProdServiceUtils.getDetailsJobName(serialNum);
            triggerKey = TriggerKey.triggerKey(detailsJobName, jobGroup);
            scheduler.pauseTrigger(triggerKey);
            logger.info("pause job: {}, {}, {} success！", jobName, queueJobName, detailsJobName);

            return true;
        } catch (SchedulerException e) {
            logger.error("pause job {} error", ProdServiceUtils.getJobName(serialNum), e);
            return false;
        }
    }

    @Override
    public boolean resumeJob(String serialNum) {
        try {
            String jobGroup = getGroupBySerialNum(serialNum);
            String jobName = ProdServiceUtils.getJobName(serialNum);
            TriggerKey triggerKey = TriggerKey.triggerKey(jobName, jobGroup);
            scheduler.resumeTrigger(triggerKey);

            // 列表子任务一起恢复
            String queueJobName = ProdServiceUtils.getQueueJobName(serialNum);
            triggerKey = TriggerKey.triggerKey(queueJobName, jobGroup);
            scheduler.resumeTrigger(triggerKey);

            // 详情子任务一起恢复
            String detailsJobName = ProdServiceUtils.getDetailsJobName(serialNum);
            triggerKey = TriggerKey.triggerKey(detailsJobName, jobGroup);
            scheduler.resumeTrigger(triggerKey);

            logger.info("resume job: {}, {}, {} success", jobName, queueJobName, detailsJobName);
            return true;
        } catch (SchedulerException e) {
            logger.error("resume job {} error", ProdServiceUtils.getJobName(serialNum), e);
            return false;
        }
    }

    @Override
    public boolean deleteJob(String serialNum) {
        try {
            String jobGroup = getGroupBySerialNum(serialNum);
            String jobName = ProdServiceUtils.getJobName(serialNum);
            JobKey jobKey = JobKey.jobKey(jobName,jobGroup);
            scheduler.deleteJob(jobKey);

            // 列表子任务一起删除
            String queueJobName = ProdServiceUtils.getQueueJobName(serialNum);
            jobKey = JobKey.jobKey(queueJobName, jobGroup);
            scheduler.deleteJob(jobKey);

            // 详情子任务一起删除
            String detailsJobName = ProdServiceUtils.getDetailsJobName(serialNum);
            jobKey = JobKey.jobKey(detailsJobName, jobGroup);
            scheduler.deleteJob(jobKey);
            logger.info("dimDel job: {}, {}, {} success", jobName, queueJobName, detailsJobName);

            // 删除任务缓存
            redisClient.del(TaskKeyConstant.getCatchSet(serialNum));
            redisClient.del(TaskKeyConstant.getQueueList(serialNum));
            redisClient.del(TaskKeyConstant.getDetailList(serialNum));
            redisClient.del(TaskKeyConstant.getDepthCatchHash(serialNum));
            redisClient.del(TaskKeyConstant.getQueueCatchSet(serialNum));
            redisClient.del(TaskKeyConstant.getUrlObjectHash(serialNum));
            redisClient.hdel(TaskKeyConstant.getTaskExecutingHash(), serialNum);

            return true;
        } catch (SchedulerException e) {
            logger.error("dimDel job {} error", ProdServiceUtils.getJobName(serialNum), e);
            return false;
        }
    }

    @Override
    public boolean restartAllJob() {
        int successCount = 0;
        int failCount = 0;
        List<UrlTask> urlTaskList = urlTaskService.selectList(new EntityWrapper().eq("execute_status", TaskStatusEnum.RUNNING.getCode()));

        for (UrlTask urlTask : urlTaskList) {
            String serialNum = urlTask.getSerialNum();
            try {
                boolean success = deleteJob(serialNum);
                if (success) {
                    success = addCronJob(serialNum);
                    if (success) {
                        logger.info("restart task success, serialNum: {}", serialNum);
                        successCount++;
                    }
                }

                if (!success) {
                    logger.warn("restart task fail, serialNum: {}", serialNum);
                    failCount++;
                }
            } catch (Exception e) {
                logger.error("restart task error, serialNum: {}", serialNum);
                failCount ++;
            }
        }

        logger.info("restart all task finish, success: {}, fail: {}", successCount, failCount);
        return true;
    }

    /**
     * 初始化 JobDetail 支持一个任务配置多个种子url
     *
     * @param jobClass
     * @param jobName
     * @param jobGroup
     * @param serialNum
     * @return org.quartz.JobDetail
     * @throws
     * @author kk.xie
     * @date 2018/10/24 20:00
     */
    private JobDetail initJobDetail(Class <? extends Job> jobClass, String jobName, String jobGroup, String serialNum){
        JobDetail jobDetail = JobBuilder.newJob(jobClass).withIdentity(jobName, jobGroup).build();
        // url请求配置
        List<UrlConfig> urlConfigList = urlConfigService.selectList(new EntityWrapper()
                .eq("serial_num",serialNum)
                .eq("status", ConfigStatusEnum.USABLE.getCode()));
        // 添加流水号
        JobDataMap jobDataMap = jobDetail.getJobDataMap();
        jobDataMap.put(Constant.SERIAL_NUM, serialNum);
        if(CollectionUtils.isNotEmpty(urlConfigList)){
            urlConfigList.forEach(urlConfig -> jobDataMap.put(urlConfig.getUrl(), urlConfig));
        }

        List<LoginAccount>  loginAccounts = loginAccountService.selectList(new EntityWrapper().eq("serial_num", serialNum));
        jobDataMap.put(Constant.LOGIN_ACCOUNT_LIST, loginAccounts);
        return jobDetail;
    }

    /**
     * 重试化 CronTrigger
     *
     * @param jobName
     * @param jobGroup
     * @param schule
     * @return org.quartz.CronTrigger
     * @throws
     * @author kk.xie
     * @date 2018/10/24 20:01
     */
    private CronTrigger initCronTrigger(String jobName, String jobGroup, String schule){
        CronScheduleBuilder scheduleBuilder = CronScheduleBuilder.cronSchedule(schule);
        CronTrigger trigger = TriggerBuilder.newTrigger()
                .withIdentity(jobName, jobGroup)
                .withSchedule(scheduleBuilder).build();
        return trigger;
    }

    /**
     * 获取任务分组
     *
     * @param serialNum
     * @return java.lang.String
     * @throws
     * @author kk.xie
     * @date 2018/10/22 16:58
     */
    private String getGroupBySerialNum(String serialNum){
        UrlTask urlTask  = urlTaskService.selectOne(new EntityWrapper().setSqlSelect("name").eq("serial_num", serialNum));
        if(urlTask != null && StringUtils.isNotEmpty(urlTask.getName())){
            return ProdServiceUtils.getJobGroupName(urlTask.getName());
        }
        return ProdServiceUtils.getJobGroupName(Constant.DEFAULT_URL_GROUP);
    }
}