package com.bxm.spider.prod.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.bxm.spider.cache.RedisClient;
import com.bxm.spider.cache.constant.TaskKeyConstant;
import com.bxm.spider.constant.json.JsonObjectParaConstant;
import com.bxm.spider.constant.processor.PretreatmentEnum;
import com.bxm.spider.constant.processor.ProcessorEnum;
import com.bxm.spider.constant.proxy.ProxyFlagEnum;
import com.bxm.spider.constant.url.UrlTypeEnum;
import com.bxm.spider.prod.model.param.TaskUrlParam;
import com.bxm.spider.prod.integration.service.DownLoadIntegrationService;
import com.bxm.spider.prod.param.DownloadParam;
import com.bxm.spider.prod.model.dao.UrlConfig;
import com.bxm.spider.prod.common.constants.Constant;
import com.bxm.spider.prod.service.UrlConfigService;
import com.bxm.spider.prod.service.UrlDealService;
import com.bxm.spider.prod.utils.ProdServiceUtils;
import com.bxm.spider.utils.NamedThreadFactory;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * URL处理服务实现
 *
 * @ClassName UrlDealServiceImpl
 * @CopyRright (c) 2018-bxm：杭州微财网络科技有限公司
 * @Author kk.xie
 * @Date 2018/10/19 19:28
 * @Version 1.0
 * @Modifier kk.xie
 * @Modify Date 2018/10/19 19:28
 **/
@Service
public class UrlDealServiceImpl implements UrlDealService {

    private Logger logger = LoggerFactory.getLogger(UrlDealServiceImpl.class);

    @Autowired
    private RedisClient redisClient;

    @Autowired
    private UrlConfigService urlConfigService;

    @Autowired
    private DownLoadIntegrationService downLoadIntegrationService;

    private final ExecutorService executorService;

    public UrlDealServiceImpl() {
        /**
         * 此线程池用于图片下载处理
         */
        this.executorService = new ThreadPoolExecutor(5,5, 0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("urlDealServiceImpl"));
    }

    @Override
    public boolean pushDetailsList(TaskUrlParam taskUrlParam) {
        JSONObject jsonObject = new JSONObject();
        if(StringUtils.isNotBlank(taskUrlParam.getJsonObject())) {
            jsonObject.put(JsonObjectParaConstant.JSONOBJECT, taskUrlParam.getJsonObject());
        }
        if(null != taskUrlParam.getPersistenceEnum() && StringUtils.isNotBlank(taskUrlParam.getPersistenceEnum().getName())) {
            jsonObject.put(JsonObjectParaConstant.PERSISTENCE, taskUrlParam.getPersistenceEnum().getName());
        }
        if(null != taskUrlParam.getProcessorEnum() && StringUtils.isNotBlank(taskUrlParam.getProcessorEnum().name())) {
            jsonObject.put(JsonObjectParaConstant.PROCESSOR, taskUrlParam.getProcessorEnum().name());
        }
        // 评论 url 类型处理
        boolean isCommentUrl = null != taskUrlParam.getType() && StringUtils.isNotBlank(taskUrlParam.getType().getValue()) && taskUrlParam.getType().equals(UrlTypeEnum.URL_COMMENT);
        if(isCommentUrl) {
            jsonObject.put(JsonObjectParaConstant.URLTYPE, UrlTypeEnum.URL_COMMENT);
        }

        String serialNum = taskUrlParam.getSerialNum();
        String originUrl = taskUrlParam.getOriginUrl();
        String detailsKey = TaskKeyConstant.getDetailList(serialNum);
        List<String> detailsUrlList = JSONArray.parseArray(taskUrlParam.getUrlList(), String.class);
        for(String url : detailsUrlList) {
            boolean isMember = redisClient.sismember(TaskKeyConstant.getCatchSet(serialNum), url);
            // 强制下载 或者 未下载过
            if (taskUrlParam.getForcedDownload() || !isMember) {
                redisClient.rpush(detailsKey, ProdServiceUtils.getSaveUrl(originUrl, url));
                redisClient.hSet(TaskKeyConstant.getUrlObjectHash(serialNum), url, jsonObject.toJSONString());
            }
        }
        return true;
    }

    @Override
    public boolean pushQueueList(TaskUrlParam taskUrlParam) {
        JSONObject jsonObject = new JSONObject();
        if(StringUtils.isNotBlank(taskUrlParam.getJsonObject())) {
            jsonObject.put(JsonObjectParaConstant.JSONOBJECT, taskUrlParam.getJsonObject());
        }
        if(null != taskUrlParam.getPersistenceEnum() && StringUtils.isNotBlank(taskUrlParam.getPersistenceEnum().getName())) {
            jsonObject.put(JsonObjectParaConstant.PERSISTENCE, taskUrlParam.getPersistenceEnum().getName());
        }
        if(null != taskUrlParam.getProcessorEnum() && StringUtils.isNotBlank(taskUrlParam.getProcessorEnum().name())) {
            jsonObject.put(JsonObjectParaConstant.PROCESSOR, taskUrlParam.getProcessorEnum().name());
        }

        String serialNum = taskUrlParam.getSerialNum();
        String originUrl = taskUrlParam.getOriginUrl();

        UrlConfig urlConfig = urlConfigService.selectOne(new EntityWrapper().eq("serial_num", serialNum).eq("url", originUrl));
        if(urlConfig == null){
            logger.warn("【推送列表队列数据异常】列表url urlConfig 不存在,taskUrlDto:{}", taskUrlParam);
            return false;
        }
        Integer depth = urlConfig.getQueueDepth() == null ? Constant.DEFAULT_DEPTH : urlConfig.getQueueDepth();

        String queueKey = TaskKeyConstant.getQueueList(serialNum);
        List<String> queueUrlList = JSONArray.parseArray(taskUrlParam.getUrlList(), String.class);
        // 种子url解析次数控制
        for(String url : queueUrlList) {
            String times = redisClient.hGet(TaskKeyConstant.getDepthCatchHash(serialNum), originUrl);
            if (times == null || Integer.valueOf(times) < depth) {
                // 列表url单次任务内不能重复
                boolean isMember = redisClient.sismember(TaskKeyConstant.getQueueCatchSet(serialNum), url);
                if (taskUrlParam.getForcedDownload() || !isMember) {
                    redisClient.hincrByOne(TaskKeyConstant.getDepthCatchHash(serialNum), originUrl);
                    // 优先级高的放在队列前端
                    if (BooleanUtils.isTrue(taskUrlParam.getProdFirst())) {
                        redisClient.lpush(queueKey, ProdServiceUtils.getSaveUrl(originUrl, url));
                    } else {
                        redisClient.rpush(queueKey, ProdServiceUtils.getSaveUrl(originUrl, url));
                    }
                    redisClient.hSet(TaskKeyConstant.getUrlObjectHash(serialNum), url, jsonObject.toJSONString());
                }
            }
        }
        return true;
    }

    @Override
    public boolean pushImageList(TaskUrlParam taskUrlParam) {
        if(null == taskUrlParam){
            logger.error("image url is null!");
            return false;
        }
        if(StringUtils.isBlank(taskUrlParam.getUrlList()) || StringUtils.isBlank(taskUrlParam.getSerialNum()) || StringUtils.isBlank(taskUrlParam.getOriginUrl())){
            logger.error("image url is empty or serialNum is empty or origin url is empty!");
            return false;
        }

        String serialNum = taskUrlParam.getSerialNum();
        String originUrl = taskUrlParam.getOriginUrl();

        //获取配置
        UrlConfig urlConfig = urlConfigService.selectOne(new EntityWrapper().eq("serial_num", serialNum).eq("url", originUrl));
        if(urlConfig == null){
            logger.warn("数据异常！图片url urlConfig 不存在,taskUrlDto:{}", taskUrlParam);
            return false;
        }
        final String cookie = urlConfig.getCookie();
        final String userAgent = urlConfig.getUserAgent();
        final String referer = urlConfig.getReferer();
        final String channel = urlConfig.getChannel();
        final String charset = urlConfig.getCharset();
        final String pretreatmentParam = urlConfig.getPretreatmentParam();
        final PretreatmentEnum pretreatmentType = StringUtils.isBlank(urlConfig.getPretreatmentType())? null:PretreatmentEnum.valueOf(urlConfig.getPretreatmentType());
        final String processorType = ProcessorEnum.IMAGE.name();
        final String persistenceType = null == taskUrlParam.getPersistenceEnum() ? urlConfig.getPersistenceType() : taskUrlParam.getPersistenceEnum().getName();
        final String object = taskUrlParam.getJsonObject();


        // 多图片 url 解析
        final String imageKey = TaskKeyConstant.getImageCatchSet(serialNum);
        List<String> imageUrlList = JSONArray.parseArray(taskUrlParam.getUrlList(), String.class);
        for(String url : imageUrlList){
            executorService.execute(() -> {
                if (StringUtils.isBlank(url)) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("the url is null,origin url is {},serialNum is {} ", originUrl, serialNum);
                    }
                    return;
                }

                //已下载去重
                boolean isMember = redisClient.sismember(imageKey, url);
                if (isMember && !taskUrlParam.getForcedDownload()) {
                    return;
                }
                // 调用下载中心
                DownloadParam downLoadParam = ProdServiceUtils.ofDownloadDto(url, originUrl, cookie, userAgent, referer, channel, serialNum, UrlTypeEnum.URL_DETAIL.getValue(),
                        processorType,charset,pretreatmentParam,pretreatmentType, ProxyFlagEnum.PROXY_CLOSE,persistenceType,object, new ArrayList<>());
                Boolean success = downLoadIntegrationService.httpDownLoadImage(downLoadParam);
                if(success){
                    // 记录到已下载队列中
                    redisClient.sadd(imageKey, url);
                    // 详情url执行成功
                    logger.info("execute image url success! serialNum: {}, redis key: {}, url: {}", serialNum, imageKey, url);
                }else{
                    logger.error("execute details url: {} error ,serialNum: {}, push to the end of the list, redis key: {}",url, serialNum, imageKey);
                }
                logger.info("【图片调度结束】url:{},存储器:{}",url, downLoadParam.getProcessorParameter().getPersistenceEnum().getName());
            });
        }
        return true;
    }
}
