package com.bxm.spider.oss.service.impl;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PutObjectResult;
import com.bxm.spider.oss.config.AliYunOssProperties;
import com.bxm.spider.oss.enums.FileTypeEnum;
import com.bxm.spider.oss.model.ImageModel;
import com.bxm.spider.oss.service.AliYunOssService;
import com.bxm.spider.utils.DateUtils;
import com.bxm.spider.utils.FileUtils;
import com.bxm.spider.utils.ImageUtils;
import com.bxm.spider.utils.UUIDUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.protocol.HTTP;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.Date;
import java.util.UUID;

@Component
public class AliYunOssServiceImpl implements AliYunOssService {

    private final static Logger log = LoggerFactory.getLogger(AliYunOssService.class);

    private final static String separator = "/";

    private final String SPIDER_PATH = "/spider/";
    private final String SPIDER_DEFAULT_PATH = "20190101666666/";

    private AliYunOssProperties aliYunOssProperties;

    @Autowired
    public AliYunOssServiceImpl(AliYunOssProperties aliYunOssProperties) {
        this.aliYunOssProperties = aliYunOssProperties;
    }

    public static void main(String[] args) {
        try {
            AliYunOssProperties properties = new AliYunOssProperties();
            AliYunOssService ossService = new AliYunOssServiceImpl(properties);
//            String str = ossService.getOssUrl("http://mtest.wstong.com/localnews_test/jpg/20190116/201812121646227797ec84d8c5-9ba6-4367-acb8-40e40923e631-spider.jpg","123");
//            String url = "http://mtest.wstong.com/localnews_test/jpg/20190116/201812121646227797ec84d8c5-9ba6-4367-acb8-40e40923e631-spider.jpg";
            String url = "http://mmbiz.qpic.cn/mmbiz/uOwWjyzdqadK9gRc2qmsW8KECUXZIicmlxnp0qe0NUdmib29uTgvxaicSyk1jUlRU8k2j8qyoicbdsNbibNjzib8fRRA/0?wx_fmt=jpeg";
            ImageModel image = ossService.getImageInfo(url,"666--",true);
            System.out.println(image);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    @Override
    public ImageModel getImageInfo(String url, String pathName,boolean isQRCodeEnable) {
        // get请求返回结果
        CloseableHttpClient httpClient = HttpClients.createDefault();
        HttpGet httpGet = new HttpGet(url);
        httpGet.addHeader(HTTP.CONTENT_TYPE, "application/json");
        // 设置请求和传输超时时间
        httpGet.setConfig(RequestConfig.custom().setSocketTimeout(10000).setConnectTimeout(10000).setConnectionRequestTimeout(10000).build());
        InputStream input = null;
        CloseableHttpResponse response = null;
        try {
            response = httpClient.execute(httpGet);
            HttpEntity httpEntity = response.getEntity();
            byte[] bytes = EntityUtils.toByteArray(httpEntity);
            // 获取图片类型加上后缀
            String suffix = "-spider";
            if (url.endsWith("gif") || url.endsWith("GIF")) {
                suffix += ".gif";
            } else if (url.endsWith("jpg") || url.endsWith("JPG")) {
                suffix += ".jpg";
            } else if (url.endsWith("jpeg") || url.endsWith("JPEG")) {
                suffix += ".jpg";
            } else if (url.endsWith("png") || url.endsWith("PNG")) {
                suffix += ".png";
            } else {
                // 无后缀等其他情况，从contentType中获取
                String typeName = getTypeFromEntity(httpEntity);

                if (StringUtils.isNotBlank(typeName)){
                    suffix += "." + typeName;
                }
            }
            String imgName = UUID.randomUUID() + suffix;
            String path = this.SPIDER_PATH;
            if (StringUtils.isNotBlank(pathName)) {
                path += pathName + "/";
            } else {
                path += SPIDER_DEFAULT_PATH;
            }

            // 判断是否二维码图片
            BufferedImage bufferedImage = ImageIO.read(new ByteArrayInputStream(bytes));
            if(null != bufferedImage) {
                boolean isQRCode = ImageUtils.isQRCode(bufferedImage);
                if(isQRCodeEnable || !isQRCode){
                    ImageModel imageModel = new ImageModel();
                    imageModel.setQrCode(isQRCode);
                    imageModel.setWidth(bufferedImage.getWidth());
                    imageModel.setHeight(bufferedImage.getHeight());
                    imageModel.setOssUrl(this.upload(bytes, path + imgName));
                    return imageModel;
                }
            }
            return null;
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            // 关闭资源
            if(null != input){
                try {
                    input.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    input = null;
                }
            }

            if(null != response){
                try {
                    response.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    response = null;
                }
            }

            if (null!= httpClient){
                try {
                    httpClient.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    httpClient = null;
                }
            }

        }
        return null;
    }
    @Override
    public String getOssUrl(String url, String pathName,String compressSuffix) {
        if(StringUtils.isBlank(url)){
            return "";
        }
        // get请求返回结果
        CloseableHttpClient httpClient = HttpClients.createDefault();
        HttpGet httpGet = new HttpGet(url);
        httpGet.addHeader(HTTP.CONTENT_TYPE, "application/json");
        // 设置请求和传输超时时间
        httpGet.setConfig(RequestConfig.custom().setSocketTimeout(10000).setConnectTimeout(10000).setConnectionRequestTimeout(10000).build());
        CloseableHttpResponse response = null;
        try {
            response = httpClient.execute(httpGet);
            HttpEntity entity = response.getEntity();
            // 获取图片类型加上后缀
            String suffix = "";
            if (url.endsWith("gif") || url.endsWith("GIF")) {
                suffix += ".gif";
            } else if (url.endsWith("jpg") || url.endsWith("JPG")) {
                suffix += ".jpg";
            } else if (url.endsWith("jpeg") || url.endsWith("JPEG")) {
                suffix += ".jpg";
            } else if (url.endsWith("png") || url.endsWith("PNG")) {
                suffix += ".png";
            } else {
                // 无后缀等其他情况，从contentType中获取
                String typeName = getTypeFromEntity(entity);
                if (StringUtils.isNotBlank(typeName)){
                    suffix += "." + typeName;
                }
            }
            String imgName = UUID.randomUUID() + suffix;
            String path = this.SPIDER_PATH;
            if (StringUtils.isNotBlank(pathName)) {
                path += pathName + "/";
            } else {
                path += SPIDER_DEFAULT_PATH;
            }
            return this.upload(entity.getContent(), path + imgName) + compressSuffix;
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            // 关闭资源
            if(null != response){
                try {
                    response.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    response = null;
                }
            }

            if (null!= httpClient){
                try {
                    httpClient.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    httpClient = null;
                }
            }
        }
        return null;
    }

    @Override
    public String getOssUrl(String url, ImageModel imageModel, String pathName) {
        // 获取图片类型加上后缀
        String suffix = "-spider."+imageModel.getImgType();;
        String imgName = UUID.randomUUID() + suffix;
        String path = this.SPIDER_PATH;
        if (StringUtils.isNotBlank(pathName)) {
            path += pathName + "/";
        } else {
            path += SPIDER_DEFAULT_PATH;
        }
        return this.upload(new ByteArrayInputStream(imageModel.getImgByte()), path + imgName);
    }


    private String getTypeFromEntity(HttpEntity entity) {
        String imageType = "";
        String contentType = entity.getContentType().getValue().toLowerCase();
        String[] type = null;
        if (StringUtils.isNotBlank(contentType)) {
            type = contentType.split("/");
        }

        if (type != null && type.length == 2) {
            imageType = type[1];
        }
        return imageType;
    }

    @Override
    public String[] upload(MultipartFile... multipartFiles) {
        String[] strings = new String[multipartFiles.length];
        try {
            int i = 0;
            for (MultipartFile multipartFile : multipartFiles) {
                String fileName = multipartFile.getOriginalFilename();
                long contentLength = multipartFile.getSize();
                //获取文件的contentType
                String contentType = multipartFile.getContentType();

                if (StringUtils.isBlank(contentType)) {
                    contentType = "text/plain";
                }

                String ossPath = generateOssPathByFile(fileName);

                PutObjectResult result = upload(multipartFile.getInputStream(), contentLength, contentType, ossPath);

                strings[i++] = getOSSFileRoot() + ossPath;
            }
        } catch (IOException e) {
            log.error("upload multipartFiles error," + e.getMessage(), e);
        }
        return strings;
    }

    @Override
    public String upload(MultipartFile multipartFile) {
        return upload(new MultipartFile[]{multipartFile})[0];
    }

    @Override
    public String upload(MultipartFile multipartFile, String path) {
        try {
            long contentLength = multipartFile.getSize();

            //获取文件的contentType
            String contentType = multipartFile.getContentType();

            if (StringUtils.isBlank(contentType)) {
                contentType = "text/plain";
            }

            String ossPath = generateOssPath(path, multipartFile.getOriginalFilename());

            PutObjectResult result = upload(multipartFile.getInputStream(), contentLength, contentType, ossPath);

            return getOSSFileRoot() + ossPath;
        } catch (Exception e) {
            log.error("upload file error:" + e.getMessage(), e);
        }
        return null;
    }

    @Override
    public String[] upload(File... files) {
        String[] uploadUrlArray = new String[files.length];
        try {
            int i = 0;
            for (File file : files) {
                //获取file.length
                long contentLength = file.length();

                //获取文件的contentType
                String suffix = FileUtils.getFileExtension(file);

                String contentType = FileTypeEnum.getContentType(suffix);
                if (StringUtils.isBlank(contentType)) {
                    contentType = "text/plain";
                }

                String fileName = file.getName();

                //生成上传的地址
                String ossPath = generateOssPathByFile(fileName);

                PutObjectResult result = upload(new FileInputStream(file), contentLength, contentType, ossPath);

                uploadUrlArray[i++] = getOSSFileRoot() + ossPath;
            }
        } catch (IOException e) {
            log.error("upload files error", e);
        }
        return uploadUrlArray;
    }

    @Override
    public String upload(File file, String path) {
        try {
            //获取file.length
            long contentLength = file.length();

            //获取文件的contentType
            String suffix = FileUtils.getFileExtension(file);
            String contentType = FileTypeEnum.getContentType(suffix);

            if (StringUtils.isBlank(contentType)) {
                contentType = "text/plain";
            }

            //生成上传的地址
            String ossPath = generateOssPath(path, file.getName());

            PutObjectResult result = upload(new FileInputStream(file), contentLength, contentType, ossPath);

            return getOSSFileRoot() + ossPath;
        } catch (Exception e) {
            log.error("upload file error:" + e.getMessage(), e);
        }
        return null;
    }

    @Override
    public String upload(byte[] data, String path) {
        try {
            long contentLength = data.length;
            //生成路径
            String contentType = FileTypeEnum.getContentType(StringUtils.substringAfterLast(path, "."));
            if (StringUtils.isBlank(contentType)) {
                contentType = "text/plain";
            }
            String ossPath = generateOssPathByPath(path);
            PutObjectResult result = upload(new ByteArrayInputStream(data), contentLength, contentType, ossPath);
            return getOSSFileRoot() + ossPath;
        } catch (Exception e) {
            log.error("字节流上传文件发生错误：" + e.getMessage(), e);
        }
        return null;
    }

    @Override
    public Boolean download(String path, String filePath) {
        try {
            OSSClient ossClient = getOssClient();
            if (path.startsWith(separator)) {
                path = path.substring(1);
            }
            path = this.aliYunOssProperties.getFolder() + separator + path;
            GetObjectRequest getObjectRequest = new GetObjectRequest(this.aliYunOssProperties.getBucket(), path);
            ossClient.getObject(getObjectRequest, new File(filePath));
            ossClient.shutdown();
            return true;
        } catch (Exception e) {
            log.error("download error", e);
            return false;
        }

    }
    @Override
    public String upload(InputStream stream, String path) {
        OSSClient client = getOssClient();
        String ossFilePath = generateOssPathByPath(path);
        String uploadPath = this.aliYunOssProperties.getFolder() + separator + ossFilePath;
        try {
            PutObjectResult result = client.putObject(this.aliYunOssProperties.getBucket(), uploadPath, stream);
            return getOSSFileRoot() + ossFilePath;
        } catch (OSSException oe) {
            oe.printStackTrace();
        } finally {
            client.shutdown();
        }
        return "";
    }
    @Override
    public String getOSSFileRoot() {
        if (StringUtils.isNotBlank(aliYunOssProperties.getCdnUrl())) {
            return aliYunOssProperties.getCdnUrl() + "/" + this.aliYunOssProperties.getFolder() + "/";
        }
        return "https://" + this.aliYunOssProperties.getBucket() + "."
                + this.aliYunOssProperties.getEndpoint() + "/"
                + this.aliYunOssProperties.getFolder() + "/";
    }

    /**
     * 根据上传的路径生成oss路径
     */
    private String generateOssPathByPath(String path) {
        String fileName = generateFileNameByPath(path);
        return generateOssPath(path, fileName);
    }

    /**
     * 根据文件名称生成oss路径
     */
    private String generateOssPathByFile(String fileName) {
        if (StringUtils.isNotBlank(fileName)) {
            fileName = UUIDUtils.nextID() + "." + generateSuffix(fileName);
        }
        if (StringUtils.isBlank(fileName)) {
            fileName = UUIDUtils.nextID();
        }
        return generateOssPath(null, fileName);
    }

    /**
     * 生成oss路径
     */
    private String generateOssPath(String path, String fileName) {
        path = handlePath(path);
        if (StringUtils.isBlank(path)) {
            path = generateDirByFileName(fileName).append(fileName).toString();
            return path;
        }

        //path为文件名
        if (!path.contains(separator)) {
            path = generateDirByFileName(fileName).append(path).toString();
            return path;
        }

        //path为目录
        if (path.endsWith(separator)) {
            String suffix = generateSuffix(fileName);
            path = path + UUIDUtils.nextID() + "." + suffix;
            return path;
        }

        return path;
    }

    /**
     * 处理path
     */
    private String handlePath(String path) {
        if (StringUtils.isBlank(path)) {
            return path;
        }

        path = path.replaceAll("\\\\", separator);
        if (path.startsWith(separator)) {
            path = path.substring(1);
        }

        return path;
    }

    /**
     * 根据path生成文件名(主要用于不能获取到文件名称)
     */
    private String generateFileNameByPath(String path) {
        path = handlePath(path);

        //path为空随机生成文件名
        if (StringUtils.isBlank(path)) {
            return UUIDUtils.nextID();
        }

        //path为文件名直接返回path
        if (!path.contains(separator)) {
            return path;
        }

        //path为目录随机生成文件名
        if (path.endsWith(separator)) {
            return UUIDUtils.nextID();
        }

        //path包含路径和文件名返回path的文件名
        return StringUtils.substringAfterLast(path, separator);
    }

    /**
     * 根据文件名设置文件上传的目录
     */
    private StringBuilder generateDirByFileName(String fileName) {
        String suffix = generateSuffix(fileName);
        String todayStr = DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date());
        return new StringBuilder().append(suffix).append(separator)
                .append(todayStr).append(separator);
    }

    /**
     * 获取文件前缀
     *
     * @param fileName
     * @return
     */
    private String generateSuffix(String fileName) {
        String suffix = FileUtils.getFilextension(fileName);
        String contentType = FileTypeEnum.getContentType(suffix);
        if (StringUtils.isBlank(contentType)) {
            suffix = "txt";
        }

        return suffix;
    }

    /**
     * 获取oss实例
     *
     * @return
     */
    private OSSClient getOssClient() {
        return new OSSClient(this.aliYunOssProperties.getEndpoint(),
                this.aliYunOssProperties.getAccessKey(),
                this.aliYunOssProperties.getAccessSecret());
    }

    /**
     * 上传文件到OSS.</br>
     *
     * @param inputStream   输入流
     * @param contentLength 文件大小
     * @param contentType   文件类型
     * @param path          保存的资源路径全称
     */
    private PutObjectResult upload(InputStream inputStream, long contentLength, String contentType, String path) {
        try {
            // 创建上传Object的Metadata
            ObjectMetadata meta = new ObjectMetadata();
            // 必须设置ContentLength
            meta.setContentLength(contentLength);
            meta.setContentType(contentType);

            OSSClient ossClient = getOssClient();
            // 上传Object.
            path = this.aliYunOssProperties.getFolder() + separator + path;

            PutObjectResult result = ossClient.putObject(this.aliYunOssProperties.getBucket(), path, inputStream, meta);

            ossClient.shutdown();
            return result;
        } catch (Exception e) {
            throw e;
        } finally {
            IOUtils.closeQuietly(inputStream);
        }
    }

}
