package com.bxm.warcar.ip.autoconfigure;

import com.bxm.warcar.ip.IpLibrary;
import com.bxm.warcar.ip.impl.AliyunIpLibrary;
import com.bxm.warcar.ip.impl.BaiduIpLibrary;
import com.bxm.warcar.ip.impl.IpipNetForDbIpLibrary;
import com.bxm.warcar.ip.impl.Ipplus360IpLibrary;
import com.bxm.warcar.ip.impl.aliyun.AliyunDnsDataServerImpl;
import com.bxm.warcar.ip.impl.aliyun.DataServer;
import com.bxm.warcar.ip.impl.aliyun.GeoipProfile;
import com.bxm.warcar.ip.impl.aliyun.OssDataServerImpl;
import com.bxm.warcar.utils.file.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.support.BeanDefinitionValidationException;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.util.ClassUtils;

import java.io.File;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;

/**
 * @author allen
 * @since V1.0.0 2017/12/14
 */
@EnableConfigurationProperties(IpLibraryConfiguration.class)
public class IpLibraryAutoConfiguration {

    @Autowired
    private IpLibraryConfiguration configuration;

    @Bean("baiduIpLibrary")
    public BaiduIpLibrary baiduIpLibrary() {
        if (!configuration.isBaiduEnable()) {
            return null;
        }
        return new BaiduIpLibrary();
    }

    // ------------------------------- aliyun start ------------------------------- //

    @Bean
    public OssDataServerImpl ossDataServer() {
        if (!configuration.isAliyunEnable()) {
            return null;
        }
        String aliyunOssDownloadUrl = configuration.getAliyunOssDownloadUrl();
        String aliyunOssDataInfoUrl = configuration.getAliyunOssDataInfoUrl();
        return new OssDataServerImpl(aliyunOssDataInfoUrl, aliyunOssDownloadUrl);
    }

    @Bean
    public AliyunDnsDataServerImpl aliyunDnsDataServer() {
        if (!configuration.isAliyunEnable()) {
            return null;
        }
        GeoipProfile profile = createProfile();
        return new AliyunDnsDataServerImpl(profile);
    }

    @Bean(name = "aliyunIpLibrary", initMethod = "init", destroyMethod = "destroy")
    public AliyunIpLibrary aliyunIpLibrary(ObjectProvider<List<DataServer>> dataServers) throws Throwable {
        if (!configuration.isAliyunEnable()) {
            return null;
        }
        GeoipProfile profile = createProfile();

        Class<? extends DataServer> dataServer = configuration.getAliyunDataServer();
        List<DataServer> dataServerList = dataServers.getIfAvailable();
        if (Objects.isNull(dataServer) || CollectionUtils.isEmpty(dataServerList)) {
            throw new BeanDefinitionValidationException("dataServer has be null!");
        }

        DataServer server = dataServerList.stream().filter(o -> (ClassUtils.getUserClass(o).equals(dataServer))).findAny()
                .orElseThrow((Supplier<Throwable>) () -> new NoSuchBeanDefinitionException(dataServer));

        return new AliyunIpLibrary(profile, server);
    }

    private GeoipProfile createProfile() {
        GeoipProfile profile = new GeoipProfile(configuration.getAliyunRegionId(), configuration.getAliyunAccessKeyId(), configuration.getAliyunSecret(), configuration.getAliyunInstanceId());
        profile.setAutoUpgrade(configuration.isAliyunAutoUpgrade());
        return profile;
    }

    // ------------------------------- aliyun end ------------------------------- //

    @Bean(name = "ipIpNetIpLibrary", initMethod = "init", destroyMethod = "destroy")
    public IpipNetForDbIpLibrary ipIpNetIpLibrary() {
        if (!configuration.isIpipNetEnable()) {
            return null;
        }
        if (StringUtils.equals(configuration.getIpipNetUseType(), IpLibraryConfiguration.USE_TYPE_REMOTE)) {

            FileComparator fileComparator = SystemFileComparatorFactory.create()
                    .fingerprintTracker(new SimpleFingerprintTracker(configuration.getIpipNetFingerprintUrl(), "ipipnet.fingerprint"))
                    .dataTracker(new SimpleDataTracker(configuration.getIpipNetDataUrl(), "ipipnet.data"))
                    .build();

            final IpipNetForDbIpLibrary library = new IpipNetForDbIpLibrary(fileComparator.getData().getPath());

            FileMonitor fileMonitor = new SystemFileMonitor(fileComparator);
            fileMonitor.addListener(library::refresh);
            fileMonitor.start();

            return library;
        }
        if (StringUtils.equals(configuration.getIpipNetUseType(), IpLibraryConfiguration.USE_TYPE_LOCALFILE)) {
            return new IpipNetForDbIpLibrary(configuration.getIpipNetFilePath());
        }
        return null;
    }

    @Bean(name = "ipplus360IpLibrary", initMethod = "init", destroyMethod = "destroy")
    public Ipplus360IpLibrary ipplus360IpLibrary() {
        if (!configuration.isIpplus360Enable()) {
            return null;
        }
        if (StringUtils.equals(configuration.getIpplus360UseType(), IpLibraryConfiguration.USE_TYPE_REMOTE)) {
            FileComparator fileComparator = SystemFileComparatorFactory.create()
                    .fingerprintTracker(new SimpleFingerprintTracker(configuration.getIpplus360FingerprintUrl(), "ipplus360.fingerprint"))
                    .dataTracker(new SimpleDataTracker(configuration.getIpplus360DataUrl(), "ipplus360.dat"))
                    .build();
            final Ipplus360IpLibrary library = new Ipplus360IpLibrary(fileComparator.getData());

            FileMonitor fileMonitor = new SystemFileMonitor(fileComparator);
            fileMonitor.addListener(library::refresh);
            fileMonitor.start();

            return library;
        }
        if (StringUtils.equals(configuration.getIpplus360UseType(), IpLibraryConfiguration.USE_TYPE_LOCALFILE)) {
            return new Ipplus360IpLibrary(new File(configuration.getIpplus360FilePath()));
        }
        return null;
    }

    @Primary
    @Bean("ipLibraries")
    public IpLibraries ipLibraries(ApplicationContext applicationContext) {
        Map<String, IpLibrary> beans = applicationContext.getBeansOfType(IpLibrary.class);
        return new IpLibraries(beans.values().toArray(new IpLibrary[0]));
    }
}
