package com.bxm.warcar.metrics.autoconfigure;

import com.bxm.warcar.metrics.ConsumeTimeMetric;
import com.bxm.warcar.metrics.CounterMetric;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

/**
 * @author allen
 * @since 1.0.0
 */
@Order(Ordered.HIGHEST_PRECEDENCE + 1)
@Aspect
@Configuration
public class MetricsAspect {

    private final ConsumeTimeMetric consumeTimeMetric;
    private final CounterMetric counterMetric;

    public MetricsAspect(ConsumeTimeMetric consumeTimeMetric, CounterMetric counterMetric) {
        this.consumeTimeMetric = consumeTimeMetric;
        this.counterMetric = counterMetric;
    }

    @Around(value = "within(@com.bxm.warcar.metrics.annotation.Metrics *)")
    public Object around(ProceedingJoinPoint point) throws Throwable {

        long start = System.currentTimeMillis();

        Object proceed = point.proceed();

        try {
            String name = getServiceName(point);
            long consumingTime = System.currentTimeMillis() - start;

            consumeTimeMetric.submit(name, consumingTime);
            counterMetric.increment(name);
        }
        catch (Exception e) {
            // Ignore
        }

        return proceed;
    }

    @AfterThrowing(value = "within(@com.bxm.warcar.metrics.annotation.Metrics *))", throwing = "throwing")
    public void afterThrowing(JoinPoint point, Throwable throwing) {
        counterMetric.decrement(getServiceName(point));
    }

    private String getServiceName(JoinPoint point) {
        return point.getSignature().toShortString();
    }
}
