package com.bxm.warcar.mq.autoconfigure;

import com.bxm.warcar.mq.Consumer;
import com.bxm.warcar.mq.Listener;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.annotation.AnnotationUtils;

import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public abstract class AbstractSubscriberHandler implements DisposableBean,  ApplicationListener<ApplicationReadyEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractSubscriberHandler.class);

    private final ConcurrentHashMap<String, Consumer> consumers = new ConcurrentHashMap<>();

    protected abstract Consumer create(String consumerId, Listener messageListener);

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        ConfigurableApplicationContext applicationContext = event.getApplicationContext();
        Map<String, Object> beans = applicationContext.getBeansWithAnnotation(Subscriber.class);
        if (MapUtils.isEmpty(beans)) {
            return;
        }
        for (Map.Entry<String, Object> entry : beans.entrySet()) {
            Object object = entry.getValue();
            register(object);
        }
    }

    @Override
    public void destroy() throws Exception {
        for (Map.Entry<String, Consumer> entry : consumers.entrySet()) {
            String key = entry.getKey();
            Consumer consumer = entry.getValue();

            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Consumer {} on shutdown", key);
            }

            consumer.shutdown();
        }
    }

    private void register(Object object) {
        if (!(object instanceof Listener)) {
            return;
        }
        if (!(object instanceof SubscriberWrapper)) {
            return;
        }

        Subscriber subscriber = AnnotationUtils.findAnnotation(object.getClass(), Subscriber.class);
        if (null == subscriber) {
            return;
        }

        SubscriberWrapper subscribe = (SubscriberWrapper) object;

        String consumerId = subscribe.getConsumerId();
        if (StringUtils.isBlank(consumerId)) {
            return;
        }

        Listener listener = (Listener) object;
        Consumer consumer = create(consumerId, listener);
        if (null == consumer) {
            return;
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Registering consumer: {} of topic: {}", consumerId, listener.getTopic());
        }
        consumer.start();

        consumers.put(consumerId, consumer);
    }

    public Map<String, Consumer> getConsumers() {
        return Collections.unmodifiableMap(consumers);
    }

    private String getProperty(Object object, String name) {
        try {
            Object value = PropertyUtils.getProperty(object, name);
            return null != value ? value.toString() : null;
        } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("getProperty: ", e);
            }
            return null;
        }
    }
}
