package com.bxm.warcar.web.interceptor;

import com.alibaba.fastjson.JSONObject;
import com.bxm.warcar.utils.IpHelper;
import com.bxm.warcar.web.autoconfigure.WebProperties;
import com.bxm.warcar.web.cache.local.LocalCache;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;
import redis.clients.jedis.JedisPool;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * <h3>Ip拦截器  </h3>
 * <p>
 * 拦截后返回结果是个空对象
 *
 * @author hcmony
 * @see com.bxm.warcar.utils.response.ResponseModel
 * </p>
 * @since V1.0.0, 2019/03/07 10:18
 */
public class IpInterceptor implements HandlerInterceptor {

	private static final Logger LOGGER = LoggerFactory.getLogger(IpInterceptor.class);
	private static final String ALL = "*";
	private static final int DURATION_TIME = 30;
	private static final String RESPONSE_MODEL = "{\"data\":null,\"success\":true,\"error\":false,\"code\":1,\"message\":\"ip\"}";

	private final LocalCache localCache;
	private final WebProperties webProperties;

	public IpInterceptor(JedisPool jedisPool) {
		this(jedisPool, new WebProperties());
	}

	public IpInterceptor(JedisPool jedisPool, WebProperties webProperties) {
		this.localCache = new LocalCache(jedisPool, DURATION_TIME, TimeUnit.MINUTES);
		this.webProperties = webProperties;
	}

	private boolean ignore(HttpServletRequest request) {
		WebProperties.Ip ip = webProperties.getIp();
		if (null == ip) {
			return false;
		}
		Map<String, Set<String>> ignoreParameters = ip.getIgnoreParameters();
		if (MapUtils.isNotEmpty(ignoreParameters)) {
			for (Map.Entry<String, Set<String>> entry : ignoreParameters.entrySet()) {
				String key = entry.getKey();
				String pvalue = request.getParameter(key);
				if (entry.getValue().contains(pvalue)) {
					return true;
				}
			}
		}
		return false;
	}

	@Override
	public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
		try {
			if (ignore(request)) {
				return true;
			}
			String ip = IpHelper.getIpFromHeader(request);

			if (StringUtils.isBlank(ip)) {
				return true;
			}

			String ipkey = IpHelper.getIpNoDotTwo(ip);
			String ipvalue = localCache.get(ipkey);

			if (StringUtils.isBlank(ipvalue)) {
				return true;
			}

			if (ALL.equals(ipvalue)) {
				write(response);
				return false;
			}

			List<String> list = JSONObject.parseArray(ipvalue, String.class);
			if (CollectionUtils.isEmpty(list)) {
				return true;
			}

			if (list.contains(ip)) {
				write(response);
				return false;
			}
		} catch (Exception e) {
			LOGGER.error(" Resolving IP exceptions !", e);
		}
		return true;
	}

	private void write(HttpServletResponse response) {
		PrintWriter writer = null;
		try {
			writer = response.getWriter();
			response.setStatus(HttpServletResponse.SC_OK);
			writer.print(RESPONSE_MODEL);
			writer.flush();
		} catch (IOException e) {
			LOGGER.error(" Resolving IP write !", e);
		} finally {
			IOUtils.closeQuietly(writer);
		}
	}


	@Override
	public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {

	}


	@Override
	public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {

	}
}
