package com.bxm.vision.data.sdk.interceptor;

import com.alibaba.fastjson.JSONObject;
import com.bxm.vision.data.sdk.autoconfigure.DotSupport;
import com.bxm.vision.data.sdk.WebProperties;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.cache.impls.redis.JedisUpdater;
import com.bxm.warcar.integration.dc.dot.DotParameter;
import com.bxm.warcar.integration.dc.dot.DotParameterFactory;
import com.bxm.warcar.integration.dc.dot.ModelTypeEnum;
import com.bxm.warcar.utils.IpHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.*;

/**
 * <h3>Ip拦截器  </h3>
 * <p>
 * 拦截后返回结果是个空对象
 *
 * @author hcmony
 * @see com.bxm.warcar.utils.response.ResponseModel
 * </p>
 * @since V1.0.0, 2019/03/07 10:18
 */
public class IpInterceptor implements HandlerInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(IpInterceptor.class);
    private static final String RESPONSE_MODEL = "\",\"success\":true,\"error\":false,\"code\":1,\"message\":\"ip\"}";
    /**
     * 示例：2BB2F4CAD877442JUKQL23EC2D9B1Y8KO
     */
    private static final String UID = "uid";
    /**
     * 示例：2bb2f4cad877442187023ec2d9b936f6
     */
    private static final String APPKEY = "appkey";

    private static final String COUNTID = "countId";
    /**
     * 示例：money-1
     */
    private static final String BUSINESS = "business";
    private final static String CLICK_HEAD = "clickMsg";
    private final static int CLICK_DB_INDEX = 11;
    private final static int CLICK_EXPIRE = 60 * 60 * 72;


    private final WebProperties webProperties;
    private final JedisFetcher sentinelJedisFetcher;
    private final JedisFetcher visionJedisFetcher;
    private final DotSupport dotSupport;
    private final JedisUpdater jedisUpdater;
    private final JedisFetcher jedisFetcher;

    public IpInterceptor(JedisFetcher sentinelJedisFetcher,
                         JedisFetcher visionJedisFetcher,
                         WebProperties webProperties,
                         DotSupport dotSupport,
                         JedisUpdater jedisUpdater,
                         JedisFetcher jedisFetcher) {
        this.webProperties = webProperties;
        this.sentinelJedisFetcher = sentinelJedisFetcher;
        this.visionJedisFetcher = visionJedisFetcher;
        this.dotSupport = dotSupport;
        this.jedisUpdater = jedisUpdater;
        this.jedisFetcher = jedisFetcher;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        try {
            // ------------------------ //
            //
            // Warning: 如果要调整这里的逻辑，请确认 adsprod-service/com.bxm.adsprod.service.ticket.CheatMonitorServiceImpl 是否需要同步更新。
            // By: Allen 2020/8/17
            //
            // ------------------------ //

            // 作弊流量过滤开关是否开启, 关闭则直接放行
            if (!isOpenFilter()) {
                return true;
            }

            // 这段逻辑保留，忽略某些具有特殊参数的请求如包含某些model_type，或者特异uid等参数
            // 当前没有用，但保留一个入口
            if (ignore(request)) {
                return true;
            }

            String ip = StringUtils.defaultIfEmpty(IpHelper.getIpFromHeader(request), "");

            String uid = StringUtils.defaultIfEmpty(getValue(request, UID), "");

            boolean cheatIp = checkIpCheat(ip);
            boolean cheatUid = checkUidCheat(uid);
            // IP,UID均未作弊则放行
            if (!cheatIp && !cheatUid) {
                return true;
            }

            String appkey = StringUtils.defaultIfEmpty(getValue(request, APPKEY), "");
            String business = StringUtils.defaultIfEmpty(getValue(request, BUSINESS), "");

            // boolean monitorPosition = checkMonitorPosition(appkey, business, cheatIp, cheatUid);
//            if (monitorPosition) {
//                return true;
//            } else {
                Map<String, String> map = Maps.newHashMap();
                banned(map, cheatIp, cheatUid);
                String countid = getValue(request, COUNTID);
                final String modeltype = getValue(request, "modeltype");
                if (StringUtils.isNotBlank(countid)) {
                    setBxmId(request, ip, appkey, business, countid,modeltype);
                    dotSupport.asyncDoGet(request, map);
                } else {
                    asyncDoGetEffect(request, map);
                }
                write(response, countid);
                return false;
//            }
        } catch (Exception e) {
            LOGGER.error(" Resolving IP exceptions !", e);
        }
        return true;
    }

    void asyncDoGetEffect( HttpServletRequest request, Map<String, String> map) {

        JSONObject json = fill(request);
        final String modeltype = getValue(request, json, "modeltype");
        if (StringUtils.isBlank(modeltype) || !"7".equals(modeltype)){
            return;
        }
        final String bxm_id = getValue(request, json, "bxm_id");

        //大数据埋点统计
        final DotParameterFactory parameterFactory = DotParameterFactory.create()
                .bxmid(bxm_id)
                .modelType(modeltype)
                .mt(ModelTypeEnum._1)
                .status(getValue(request, json, "status"))
                .sh(json.getInteger("sh"))
                .sw(json.getInteger("sw"))
                .ts(json.getLong("ts"))
                .locaurl(getValue(request, json, "locaurl"))
                .referrer(request.getHeader("referrer"));

        final String result = jedisFetcher.fetchWithSelector(() -> KeyBuilder.build(CLICK_HEAD, bxm_id), String.class, CLICK_DB_INDEX);
        if (StringUtils.isNotBlank(result)) {
            String[] results = result.split(":");
            try {
                parameterFactory.ip(results[0]);
                parameterFactory.appkey(results[1]);
                parameterFactory.business(results[2]);
                parameterFactory.activityId(results[3]);
                parameterFactory.awardId(results[4]);
                parameterFactory.uid(results[5]);
                parameterFactory.assetsId(results[6]);
                parameterFactory.ua(results[8]);

            } catch (Exception e) {
                LOGGER.error(e.getMessage());
            }
        }

        final DotParameter parameter = parameterFactory.build();
        parameter.put("fmModelId", request.getParameter("fmModelId"));
        parameter.put("banned", map.get("banned"));
        parameter.put("did", request.getParameter("did"));
        dotSupport.asyncDoGetEffect(parameter);
    }

    String getValue(HttpServletRequest request, JSONObject jsonObject, String key){
        final String value = request.getParameter(key);
        if (org.apache.commons.lang3.StringUtils.isNotBlank(value)){
            return value;
        }
        return jsonObject.getString(key);
    }
    /**
     * 填充参数
     * @param request
     */
    JSONObject fill(HttpServletRequest request){
        //从请求头中获取参数
        JSONObject jsonObject = new JSONObject();
        StringBuilder sb = new StringBuilder();
        try (BufferedReader reader = request.getReader();) {
            char[] buff = new char[1024];
            int len;
            while ((len = reader.read(buff)) != -1) {
                sb.append(buff, 0, len);
            }
        } catch (IOException e) {
            LOGGER.error("fill error ", e);
            return jsonObject;
        }
        String params = sb.toString();

        //把为空的参数再赋值
        if (org.apache.commons.lang3.StringUtils.isNotBlank(params)) {
            jsonObject = JSONObject.parseObject(params);
        }
        return jsonObject;
    }

    private void setBxmId(HttpServletRequest request, String ip, String appkey, String business, String countid,String modeltype) {
        if (!"7".equals(modeltype)) {
            return;
        }
        String value = KeyBuilder.build(ip,
                appkey,
                business,
                getValue(request, "activityid"),
                getValue(request, "preid"),
                getValue(request, "uid"),
                getValue(request, "assetsId"),
                getValue(request, "appos"),
                getValue(request, "ua"));

        jedisUpdater.updateWithSelector(() -> KeyBuilder.build(CLICK_HEAD, countid), value, CLICK_EXPIRE, CLICK_DB_INDEX);
    }


    private boolean ignore(HttpServletRequest request) {
        WebProperties.Ip ip = webProperties.getIp();
        if (null == ip) {
            return false;
        }
        Map<String, Set<String>> ignoreParameters = ip.getIgnoreParameters();
        if (MapUtils.isNotEmpty(ignoreParameters)) {
            for (Map.Entry<String, Set<String>> entry : ignoreParameters.entrySet()) {
                String key = entry.getKey();
                String pvalue = request.getParameter(key);
                if (entry.getValue().contains(pvalue)) {
                    return true;
                }
            }
        }
        return false;
    }


    private boolean checkMonitorPosition(String appkey, String business, boolean cheatIp, boolean cheatUid) {
        String position = appkey + (business.replaceAll("money|ad", ""));
        if (StringUtils.isBlank(position)) {
            return false;
        }

        //{"map":{"BACK_TICKET":[true],"ACCOUNT_TYPE":[],"TICKET_ID":[],"MONITOR_GRANULARITY":["UID","IP"]},"processMode":["MONITOR_GRANULARITY","BACK_TICKET"],"ticketNotLimit":false}
        PositionMonitorBo positionMonitorBo = sentinelJedisFetcher.hfetch(() -> "AD:CHEAT:MONITOR:POSITION", position, PositionMonitorBo.class);
        if (Objects.nonNull(positionMonitorBo)) {
            Set<String> monitorGranularitySet = (Set<String>) positionMonitorBo.getSet(MonitorEnum.MONITOR_GRANULARITY);
            if (monitorGranularitySet.contains(PositionMonitorBo.IP) && cheatIp) {
                return true;
            }
            if (monitorGranularitySet.contains(PositionMonitorBo.UID) && cheatUid) {
                return true;
            }
        }

        PositionMonitorBo all = sentinelJedisFetcher.hfetch(() -> "AD:CHEAT:MONITOR:POSITION", "ALL", PositionMonitorBo.class);
        if (Objects.nonNull(all)) {
            Set<MonitorEnum> processMode = all.getProcessMode();
            // 特殊redis key 处理
            if (CollectionUtils.isEmpty(processMode)) {
                return false;
            } else if (processMode.size() == 1 && processMode.contains(MonitorEnum.MONITOR_GRANULARITY)) {
                return false;
            } else {
                Set<String> monitorGranularitySet = (Set<String>) all.getSet(MonitorEnum.MONITOR_GRANULARITY);
                if (monitorGranularitySet.contains(PositionMonitorBo.IP) && cheatIp) {
                    return true;
                }
                if (monitorGranularitySet.contains(PositionMonitorBo.UID) && cheatUid) {
                    return true;
                }
            }
        }
        return false;
    }

    private boolean checkUidCheat(String uid) {
        return StringUtils.isNotBlank(uid) && Objects.nonNull(visionJedisFetcher.fetch(() -> StringUtils.join(new String[]{"CHEAT", "UID", uid}, ":"), Object.class));
    }

    private boolean checkIpCheat(String ip) {
        return StringUtils.isNotBlank(ip) && Objects.nonNull(visionJedisFetcher.fetch(() -> StringUtils.join(new String[]{"CHEAT", "IP", ip}, ":"), Object.class));
    }

    private String getValue(HttpServletRequest request, String key) {
        String value = request.getParameter(key);
        return value;
    }


    private boolean isOpenFilter() {
        Boolean isOpenFilter = sentinelJedisFetcher.fetch(() -> "AD:CHEAT:FLOW:FILTER:OPTION", Boolean.class);
        if (Boolean.TRUE.equals(isOpenFilter)) {
            return true;
        }
        return false;
    }

    private void write(HttpServletResponse response, String data) {
        PrintWriter writer = null;
        try {
            writer = response.getWriter();
            response.setStatus(HttpServletResponse.SC_OK);
            writer.print("{\"data\":\"" + data + RESPONSE_MODEL);
            writer.flush();
        } catch (IOException e) {
            LOGGER.error(" Resolving IP write !", e);
        } finally {
            IOUtils.closeQuietly(writer);
        }
    }


    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {

    }


    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {

    }

    private void banned(Map<String, String> map, boolean cheatIp, boolean cheatUid) {
        if (cheatIp) {
            map.put("banned", "1");
        }
        if (cheatUid) {
            map.put("banned", "2");
        }
        if (cheatIp && cheatUid) {
            map.put("banned", "3");
        }
    }


    static class PositionMonitorBo implements Serializable {
        private static final String IP = "IP";
        private static final String UID = "UID";
        private static final long serialVersionUID = -2150559929809588794L;

        private boolean ticketNotLimit;

        /**
         * 处理策略
         */
        private final Set<MonitorEnum> processMode = new HashSet();

        private final Map<MonitorEnum, Set> map = new HashedMap();

        public PositionMonitorBo() {
            MonitorEnum[] monitorEnums = MonitorEnum.values();
            for (MonitorEnum monitorEnum : monitorEnums) {
                map.put(monitorEnum, new HashSet());
            }
        }


        public Set getSet(MonitorEnum monitorEnum) {
            return map.get(monitorEnum);
        }

        public void removeSet(MonitorEnum monitorEnum) {
            map.remove(monitorEnum);
            processMode.remove(monitorEnum);
        }

        public void removeAllMonitorSet() {
            MonitorEnum[] values = MonitorEnum.values();
            for (MonitorEnum value : values) {
                if (MonitorEnum.MONITOR_GRANULARITY.equals(value)) {
                    continue;
                }
                map.remove(value);
                processMode.remove(value);
            }
        }

        public void removeAllSet() {
            MonitorEnum[] values = MonitorEnum.values();
            for (MonitorEnum value : values) {
                map.remove(value);
                processMode.remove(value);
            }
        }

        public boolean add(MonitorEnum monitorEnum, Object obj) {
            if (Objects.isNull(obj)) {
                return true;
            }

            Set set = map.get(monitorEnum);
            if (Objects.isNull(set)) {
                set = new HashSet();
            }
            processMode.add(monitorEnum);
            set.add(obj);

            return true;
        }

        public boolean addAll(MonitorEnum monitorEnum, Collection collection) {
            if (CollectionUtils.isEmpty(collection)) {
                return true;
            }

            Set set = map.get(monitorEnum);
            if (Objects.isNull(set)) {
                set = new HashSet();
            }
            processMode.add(monitorEnum);
            set.addAll(collection);
            return true;
        }

        public boolean isTicketNotLimit() {
            return ticketNotLimit;
        }

        public void setTicketNotLimit(boolean ticketNotLimit) {
            this.ticketNotLimit = ticketNotLimit;
        }

        public Set<MonitorEnum> getProcessMode() {
            return processMode;
        }

        public Map<MonitorEnum, Set> getMap() {
            return map;
        }


    }

    enum MonitorEnum {
        /**
         * 广告券监控策略
         */
        TICKET_ID,
        /**
         * 结算类型
         */
        ACCOUNT_TYPE,
        /**
         * 备用券
         */
        BACK_TICKET,
        /**
         * 监控粒度
         */
        MONITOR_GRANULARITY
    }
}
