package com.bxm.warcar.cache.impls.redis;

import com.bxm.warcar.cache.*;
import com.bxm.warcar.cache.serialization.JSONSerialization;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

public class JedisFetcher implements Fetcher {

    private final JedisPool jedisPool;
    private final JSONSerialization serialization;

    public JedisFetcher(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
        this.serialization = new JSONSerialization();
    }

    @Override
    public <T> T fetch(KeyGenerator keyGenerator, DataExtractor<T> dataExtractor, Class<T> cls) {
        return fetch(keyGenerator, dataExtractor, cls, 0);
    }

    @Override
    public <T> T fetch(KeyGenerator keyGenerator, DataExtractor<T> dataExtractor, Class<T> cls, int expireTimeInSecond) {
        if (null == keyGenerator) {
            throw new NullPointerException("keyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();

            String value = jedis.get(key);

            if (null != value) {
                return serialization.deserialize(value, cls);
            }

            T fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
            }

            if (null != fromStore) {
                jedis.set(key, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0)
                    jedis.expire(key, expireTimeInSecond);
            }

            return fromStore;
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }
}
