package com.bxm.warcar.cache.impls.redis;

import com.bxm.warcar.cache.RedisLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.params.SetParams;

import java.util.Collections;

/**
 * <h3>
 * 基于redis的分布式锁
 * 1,非重入锁,如需要重入功能，请使用redisson
 * 2,tryLock 与unlock key 与 pid 保持一致
 * </h3>
 *
 * @author hcmony
 * @since V1.0.0, 2019/10/28 22:00
 */
public class DistributedRedisLock implements RedisLock {

    private static final Logger LOGGER = LoggerFactory.getLogger(DistributedRedisLock.class);

    private static final String LOCK_SUCCESS = "OK";
    private static final Long UNLOCK_SUCCESS = 1L;
    private static final String SCRIPT = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";
    private final JedisPool jedisPool;
    private static final int DEFAULT_EXPIRETIME = 1000 * 30;

    public DistributedRedisLock(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    /**
     * 加锁
     *
     * @param key
     * @param rid
     * @param expireTime 这里是毫秒
     * @return
     */
    @Override
    public boolean tryLock(String key, String rid, int expireTime) {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            String result = jedis.set(key, rid, SetParams.setParams().nx().px(expireTime));
            return LOCK_SUCCESS.equals(result);
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return false;
    }

    /**
     * 加锁,默认失效60秒
     *
     * @param key
     * @param rid
     * @return
     */
    @Override
    public boolean tryLock(String key, String rid) {
        return tryLock(key, rid, DEFAULT_EXPIRETIME);
    }

    /**
     * 解锁
     *
     * @param key
     * @param rid
     * @return
     */
    @Override
    public boolean unLock(String key, String rid) {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            Object result = jedis.eval(SCRIPT, Collections.singletonList(key), Collections.singletonList(rid));
            return UNLOCK_SUCCESS.equals(result);

        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return false;
    }
}
