package com.bxm.warcar.cache.impls.redis;

import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.ScanParams;
import redis.clients.jedis.ScanResult;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;

/**
 * @author allen
 * @date 2017-12-04
 */
public class JedisCounter implements Counter {

    private final JedisPool jedisPool;

    public JedisCounter(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    @Override
    public Object getClientOriginal() {
        return jedisPool;
    }

    @Override
    public Long incrementAndGet(KeyGenerator keyGenerator) {
        return incrementAndGet(keyGenerator, 0);
    }

    @Override
    public Long incrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond) {
        return incrementAndGet(keyGenerator, expireTimeInSecond, incr -> true);
    }

    @Override
    public Long incrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond, Predicate<Long> updateExpireTime) {
        Objects.requireNonNull(updateExpireTime);
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Long incr = jedis.incr(key);
            if (updateExpireTime.test(incr)) {
                setExpire(jedis, key, expireTimeInSecond);
            }
            return incr;
        }
    }

    @Override
    public Long incrementByAndGet(KeyGenerator keyGenerator, long inc) {
        return incrementByAndGet(keyGenerator, inc, 0);
    }

    @Override
    public Long incrementByAndGet(KeyGenerator keyGenerator, long inc, int expireTimeInSecond) {
        return incrementByAndGet(keyGenerator, inc, expireTimeInSecond, incr -> true);
    }

    @Override
    public Long incrementByAndGet(KeyGenerator keyGenerator, long inc, int expireTimeInSecond, Predicate<Long> updateExpireTime) {
        Objects.requireNonNull(updateExpireTime);
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Long incr = jedis.incrBy(key, inc);
            if (updateExpireTime.test(incr)) {
                setExpire(jedis, key, expireTimeInSecond);
            }
            return incr;
        }
    }

    @Override
    public Double incrFloatByAndGet(KeyGenerator keyGenerator, double inc) {
        return incrFloatByAndGet(keyGenerator, inc, 0);
    }

    @Override
    public Double incrFloatByAndGet(KeyGenerator keyGenerator, double inc, int expireTimeInSecond) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Double incr = jedis.incrByFloat(key, inc);
            setExpire(jedis, key, expireTimeInSecond);
            return incr;
        }
    }

    @Override
    public Long decrementAndGet(KeyGenerator keyGenerator) {
        return decrementAndGet(keyGenerator, 0);
    }

    @Override
    public Long decrementAndGet(KeyGenerator keyGenerator, int expireTimeInSecond) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Long decr = jedis.decr(key);
            setExpire(jedis, key, expireTimeInSecond);
            return decr;
        }
    }

    @Override
    public Long decrementByAndGet(KeyGenerator keyGenerator, long dec) {
        return decrementByAndGet(keyGenerator, dec, 0);
    }

    @Override
    public Long decrementByAndGet(KeyGenerator keyGenerator, long dec, int expireTimeInSecond) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Long decr = jedis.decrBy(key, dec);
            setExpire(jedis, key, expireTimeInSecond);
            return decr;
        }
    }

    @Override
    public void set(KeyGenerator keyGenerator, long value) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            jedis.set(key, String.valueOf(value));
        }
    }

    @Override
    public void expire(KeyGenerator keyGenerator, int seconds) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            setExpire(jedis, key, seconds);
        }
    }

    @Override
    public Long get(KeyGenerator keyGenerator) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            return TypeHelper.castToLong(jedis.get(key));
        }
    }

    @Override
    public Long hincrementAndGet(KeyGenerator keyGenerator, String field) {
        return hincrementAndGet(keyGenerator, field, 0);
    }

    @Override
    public Long hincrementAndGet(KeyGenerator keyGenerator, String field, int expireTimeInSecond) {
        return hincrementByAndGet(keyGenerator, field, 1, expireTimeInSecond);
    }

    @Override
    public Long hincrementAndGet(KeyGenerator keyGenerator, String field, int expireTimeInSecond, Predicate<Long> updateExpireTime) {
        return hincrementByAndGet(keyGenerator, field, 1, expireTimeInSecond, updateExpireTime);
    }

    @Override
    public Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc) {
        return hincrementByAndGet(keyGenerator, field, inc, 0);
    }

    @Override
    public Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc, int expireTimeInSecond) {
        return hincrementByAndGet(keyGenerator, field, inc, expireTimeInSecond, h -> true);
    }

    @Override
    public Long hincrementByAndGet(KeyGenerator keyGenerator, String field, long inc, int expireTimeInSecond, Predicate<Long> updateExpireTime) {
        Objects.requireNonNull(updateExpireTime);
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Long hincr = jedis.hincrBy(key, field, inc);
            if (updateExpireTime.test(hincr)) {
                setExpire(jedis, key, expireTimeInSecond);
            }
            return hincr;
        }
    }

    @Override
    public Double hincrFloatByAndGet(KeyGenerator keyGenerator, String field, double inc) {
        return hincrFloatByAndGet(keyGenerator, field, inc, 0);
    }

    @Override
    public Double hincrFloatByAndGet(KeyGenerator keyGenerator, String field, double inc, int expireTimeInSecond) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Double decr = jedis.hincrByFloat(key, field, inc);
            setExpire(jedis, key, expireTimeInSecond);
            return decr;
        }
    }

    @Override
    public Long hget(KeyGenerator keyGenerator, String field) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            String v = jedis.hget(key, field);
            return NumberUtils.toLong(v, 0);
        }
    }

    @Override
    public Map<String, Long> hgetall(KeyGenerator keyGenerator) {
        String key = getKey(keyGenerator);
        try (Jedis jedis = jedisPool.getResource()) {
            Map<String, Long> rst = Maps.newHashMap();
            final String startCursor = "0";
            String cursor = startCursor;
            do {
                ScanResult<Map.Entry<String, String>> hscan = jedis.hscan(key, cursor, new ScanParams().count(10000));
                cursor = hscan.getCursor();
                List<Map.Entry<String, String>> result = hscan.getResult();
                if (CollectionUtils.isNotEmpty(result)) {
                    for (Map.Entry<String, String> entry : result) {
                        String k = entry.getKey();
                        String v = entry.getValue();
                        rst.put(k, NumberUtils.toLong(v));
                    }
                }
            } while (!cursor.equals(startCursor));
            return rst;
        }
    }

    private void setExpire(Jedis jedis, String key, int expireTimeInSecond) {
        if (expireTimeInSecond > 0) {
            jedis.expire(key, (long) expireTimeInSecond);
        }
    }

    private String getKey(KeyGenerator keyGenerator) {
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }
        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        return key;
    }
}
