package com.bxm.warcar.xcache.fetchers;

import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.serialization.JSONSerialization;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.Target;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.KeyValue;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.keyvalue.DefaultKeyValue;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.*;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 基于Redis实现
 * @author allen
 * @date 2019/1/7
 * @since 1.0.0
 */
public class RedisFetcher implements Fetcher {

    private final JedisPool jedisPool;
    private final JSONSerialization serialization;

    public RedisFetcher(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
        this.serialization = new JSONSerialization();
    }

    @Override
    public <T> T fetch(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        DataExtractor<T> dataExtractor = target.getDataExtractor();
        int selector = target.getSelector();
        int expireTimeInSecond = target.getExpireTimeInSecond();
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            String value = jedis.get(key);

            if (null != value) {
                return serialization.deserialize(value, cls);
            }

            T fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
                expireTimeInSecond = dataExtractor.updateExpireTimeInSecond(fromStore, expireTimeInSecond);
            }

            if (null != fromStore) {
                jedis.set(key, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0) {
                    jedis.expire(key, expireTimeInSecond);
                }
            }

            return fromStore;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> List<T> fetchToList(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        DataExtractor<List<T>> dataExtractor = target.getListDataExtractor();
        int selector = target.getSelector();
        int expireTimeInSecond = target.getExpireTimeInSecond();
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            String value = jedis.get(key);

            if (null != value) {
                return serialization.deserializeList(value, cls);
            }

            List<T> fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
                expireTimeInSecond = dataExtractor.updateExpireTimeInSecond(fromStore, expireTimeInSecond);
            }

            if (null != fromStore) {
                jedis.set(key, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0) {
                    jedis.expire(key, expireTimeInSecond);
                }
            }

            return fromStore;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> T hfetch(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        DataExtractor<T> dataExtractor = target.getDataExtractor();
        int selector = target.getSelector();
        int expireTimeInSecond = target.getExpireTimeInSecond();
        String field = target.getField();
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        if (StringUtils.isBlank(field)) {
            throw new NullPointerException("field");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            String value = jedis.hget(key, field);

            if (null != value) {
                return serialization.deserialize(value, cls);
            }

            T fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
                expireTimeInSecond = dataExtractor.updateExpireTimeInSecond(fromStore, expireTimeInSecond);
            }

            if (null != fromStore) {
                jedis.hset(key, field, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0) {
                    jedis.expire(key, expireTimeInSecond);
                }
            }

            return fromStore;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> List<T> hfetchToList(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        DataExtractor<List<T>> dataExtractor = target.getListDataExtractor();
        int selector = target.getSelector();
        int expireTimeInSecond = target.getExpireTimeInSecond();
        String field = target.getField();

        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        if (StringUtils.isBlank(field)) {
            throw new NullPointerException("field");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            String value = jedis.hget(key, field);

            if (null != value) {
                return serialization.deserializeList(value, cls);
            }

            List<T> fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
                expireTimeInSecond = dataExtractor.updateExpireTimeInSecond(fromStore, expireTimeInSecond);
            }

            if (null != fromStore) {
                jedis.hset(key, field, serialization.serialize(fromStore));
                if (expireTimeInSecond > 0) {
                    jedis.expire(key, expireTimeInSecond);
                }
            }

            return fromStore;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> Map<String, T> hfetchall(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        DataExtractor<Map<String, T>> dataExtractor = target.getMapDataExtractor();
        int selector = target.getSelector();
        int expireTimeInSecond = target.getExpireTimeInSecond();

        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            Map<String, T> rst = Maps.newHashMap();
            final String startCursor = "0";
            String cursor = startCursor;
            ScanParams scanParams = new ScanParams().count(target.getBatchSize());
            do {
                ScanResult<Map.Entry<String, String>> hscan = jedis.hscan(key, cursor, scanParams);
                cursor = hscan.getCursor();
                List<Map.Entry<String, String>> result = hscan.getResult();
                if (CollectionUtils.isNotEmpty(result)) {
                    for (Map.Entry<String, String> entry : result) {
                        String k = entry.getKey();
                        String v = entry.getValue();
                        rst.put(k, serialization.deserialize(v, cls));
                    }
                }
            } while (!cursor.equals(startCursor));

            if (MapUtils.isNotEmpty(rst)) {
                return rst;
            }

            Map<String, T> fromStore = null;
            if (null != dataExtractor) {
                fromStore = dataExtractor.extract();
                expireTimeInSecond = dataExtractor.updateExpireTimeInSecond(fromStore, expireTimeInSecond);
            }

            if (null != fromStore) {
                Set<Map.Entry<String, T>> entries = fromStore.entrySet();
                for (Map.Entry<String, T> entry : entries) {
                    jedis.hset(key, entry.getKey(), serialization.serialize(entry.getValue()));
                }
                if (expireTimeInSecond > 0) {
                    jedis.expire(key, expireTimeInSecond);
                }
            }

            return fromStore;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> Set<T> sfetchall(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        int selector = target.getSelector();
        if (null == keyGenerator) {
            throw new NullPointerException("keyGenerator");
        }
        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            Set<T> rs = Sets.newHashSet();
            String cursor = "0";
            ScanParams scanParams = new ScanParams().count(target.getBatchSize());
            do {
                ScanResult<String> scanResult = jedis.sscan(key, cursor, scanParams);
                cursor = scanResult.getCursor();
                List<String> result = scanResult.getResult();
                if (CollectionUtils.isNotEmpty(result)) {
                    for (String s : result) {
                        rs.add(serialization.deserialize(s, cls));
                    }
                }
            } while (!"0".equals(cursor));
            return rs;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    /**
     * 不会从 DataExtractor 获取数据，如果 Redis 中没有那则直接返回。
     *
     * @param target 目标
     * @param <T>
     * @return
     */
    @Override
    public <T> List<T> zfetch(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        int selector = target.getSelector();
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            List<T> rst = Lists.newArrayList();
            String cursor = "0";
            ScanParams scanParams = new ScanParams().count(target.getBatchSize());
            do {
                ScanResult<Tuple> zscan = jedis.zscan(key, cursor, scanParams);
                cursor = zscan.getCursor();
                List<Tuple> result = zscan.getResult();
                if (CollectionUtils.isNotEmpty(result)) {
                    for (Tuple t : result) {
                        rst.add(serialization.deserialize(t.getElement(), cls));
                    }
                }
            } while (!"0".equals(cursor));
            return rst;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> List<KeyValue> zfetchall(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        Class<T> cls = target.getCls();
        int selector = target.getSelector();
        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }
            List<KeyValue> rs = Lists.newArrayList();
            String cursor = "0";
            ScanParams scanParams = new ScanParams().count(target.getBatchSize());
            do {
                ScanResult<Tuple> zscan = jedis.zscan(key, cursor, scanParams);
                cursor = zscan.getCursor();
                List<Tuple> result = zscan.getResult();
                if (CollectionUtils.isNotEmpty(result)) {
                    for (Tuple t : result) {
                        DefaultKeyValue kv = new DefaultKeyValue();
                        kv.setKey(serialization.deserialize(t.getElement(), cls));
                        kv.setValue(t.getScore());
                        rs.add(kv);
                    }
                }
            } while (!"0".equals(cursor));

            return rs;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> long distinct(Target<T> target) {
        Preconditions.checkNotNull(target);
        KeyGenerator keyGenerator = target.getKeyGenerator();
        int selector = target.getSelector();

        if (null == keyGenerator) {
            throw new NullPointerException("getKeyGenerator");
        }

        String key = keyGenerator.generateKey();
        if (StringUtils.isBlank(key)) {
            throw new NullPointerException("key");
        }

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (selector >= 0) {
                jedis.select(selector);
            }

            return jedis.pfcount(key);
        }
        finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public <T> T getClientOriginal() {
        return (T) this.jedisPool;
    }
}
