package com.bxm.warcar.configure.loader;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.IOException;

/**
 * 如果远程服务无法获取到数据，则使用本地缓存的。获取成功时会将副本保存到本地缓存。
 *
 * @author huxiao
 * @date 2024/4/22
 * @since 1.0.0
 */
@Slf4j
public abstract class AbstractCacheEnvironmentLoader extends AbstractEnvironmentLoader {

    private File cacheDir;
    /**
     * 是否启用缓存
     */
    private final boolean enableCache;

    public AbstractCacheEnvironmentLoader(String address, String[] keys) {
        super(address, keys);
        this.enableCache = createCacheDirIfNecessary();
    }

    protected abstract byte[] readSource0(String key);

    @Override
    protected byte[] readSource(String key) {
        if (!enableCache) {
            return readSource0(key);
        }
        byte[] source0 = readSource0(key);
        if (source0 != null) {
            this.writeCacheFile(key, source0);
            return source0;
        } else {
            return readCacheFile(key);
        }
    }

    private byte[] readCacheFile(String key) {
        File cacheFile = new File(cacheDir, key);
        try {
            if (cacheFile.exists()) {
                byte[] bytes = FileUtils.readFileToByteArray(cacheFile);
                log.info("Successful read configure data length {} byte from file: {}", bytes.length, cacheFile.getAbsolutePath());
                return bytes;
            }
        } catch (IOException e) {
            log.error("readCacheFile: ", e);
        }
        return null;
    }

    private void writeCacheFile(String key, byte[] source) {
        File cacheFile = new File(cacheDir, key);
        try {
            FileUtils.writeByteArrayToFile(cacheFile, source);
            log.info("Successful write configure to file: {}", cacheFile.getAbsolutePath());
        } catch (IOException e) {
            log.error("writeCacheFile: ", e);
        }
    }

    private boolean createCacheDirIfNecessary() {
        String tmpdir = System.getProperty("user.home") + File.separator + ".warcar" + File.separator + "configure" + File.separator;
        this.cacheDir = new File(tmpdir);
        if (!cacheDir.exists()) {
            try {
                if (!cacheDir.mkdirs()) {
                    throw new IOException("无法创建目录：" + cacheDir.getAbsolutePath());
                }
            } catch (SecurityException | IOException e) {
                log.warn("无法创建目录：" + cacheDir.getAbsolutePath(), e);
                return false;
            }
        } else {
            if (!cacheDir.isDirectory()) {
                log.warn("文件路径 {} 已经存在，且不是目录。所以不支持配置缓存本地功能。", cacheDir.getAbsolutePath());
                return false;
            }
        }
        return true;
    }
}
