/*
 * Copyright 2016 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * textile.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it only
 * in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */

package com.bxm.warcar.configure.spring;

import com.bxm.warcar.configure.EnvironmentLoader;
import com.bxm.warcar.configure.loader.EnvironmentLoaderFactory;
import com.bxm.warcar.configure.update.UpdateEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.context.event.ApplicationEnvironmentPreparedEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertiesPropertySource;

import java.util.List;
import java.util.Properties;

/**
 * <h3>SpringBoot environment prepared listener.</h3>
 * <p>
 *     First step: you need set properties about configure to application.properties
 * </p>
 * <pre>
 *     application.properties
 *
 *     configure.address=zookeeper://127.0.0.1
 *     configure.key=/config/application.yml,/config/application.properties
 * </pre>
 * <p>
 *     Second step: Add this listener to SpringApplication
 * </p>
 * <pre>
 * <code>@</code>SpringBootConfiguration
 * public class SpringBootEnvironmentLoaderTest {
 *
 *     public static void main(String[] args) {
 *          SpringApplication application = new SpringApplication(SpringBootEnvironmentLoaderTest.class);
 *          application.addListeners(new SpringBootEnvironmentListener());
 *          application.run(args);
 *     }
 * }
 * </pre>
 * <p>Or use ApplicationListener:</p>
 * <pre>
 *     file: classpath:/META-INF/spring.factories
 *     content:
 *     org.springframework.context.ApplicationListener=\
 *      com.bxm.warcar.configure.spring.SpringBootEnvironmentListener
 * </pre>
 *
 * @author allen
 * @since V1.0.0 2017/12/21
 */
public final class SpringBootEnvironmentListener implements ApplicationListener<ApplicationEnvironmentPreparedEvent>, DisposableBean, ApplicationContextAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(SpringBootEnvironmentListener.class);

    private static final String ADDRESS = "configure.address";
    private static final String KEY = "configure.key";

    private EnvironmentLoader environmentLoader;

    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void onApplicationEvent(ApplicationEnvironmentPreparedEvent event) {
        ConfigurableEnvironment environment = event.getEnvironment();

        boolean contains = environment.containsProperty(ADDRESS);
        if (!contains) {
            return;
        }
        Object address = environment.getProperty(ADDRESS);
        Object key = environment.getProperty(KEY);
        if (null == address || null == key) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("Environment property '{}','{}' must not be null.", ADDRESS, KEY);
            }
            return;
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Loading properties '{}' from {}", key, address);
        }

        environmentLoader = EnvironmentLoaderFactory.create(address.toString(), key.toString(),
                (properties -> {
                    this.refresh(environment, properties);

                    if (null != applicationContext) {
                        applicationContext.publishEvent(new UpdateEvent(this));
                    }
                }));
        if (null == environmentLoader) {
            return;
        }
        List<Properties> properties = environmentLoader.loadProperties();

        this.refresh(environment, properties);
    }

    @Override
    public void destroy() throws Exception {
        this.closeLoader(environmentLoader);
    }

    private void closeLoader(EnvironmentLoader environmentLoader) {
        try {
            environmentLoader.destroy();
        } catch (Exception ignore) {
            ;
        }
    }

    private synchronized void refresh(ConfigurableEnvironment environment, List<Properties> properties) {
        if (null != properties) {
            MutablePropertySources propertySources = environment.getPropertySources();
            for (int i = 0; i < properties.size(); i++) {
                // 在指定profile情况下，可能会存在default profile和指定profile配置项重复，而Spring内部会从sources列表的索引0开始查找，查到后即返回。
                // 所以远程配置为了覆盖本地配置，需要添加到本地配置之前。
                // 这里可以添加到 random 位置后，但为了兼容性，可能后面的版本会移除 random，所以直接添加在头部。
                String name = "userRemoteDefinedProperties-" + i;
                propertySources.addFirst(new PropertiesPropertySource(name, properties.get(i)));

                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Load successful on PropertySource '{}'", name);
                }
            }
        }
    }
}
