package com.bxm.warcar.datasync.source.redis;

import com.bxm.warcar.datasync.source.SourceController;
import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

/**
 * @author allen
 * @date 2019/4/2
 * @since 1.0.0
 */
public class RedisSourceController implements SourceController {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisSourceController.class);
    public static final String DEFAULT_PREFIX_KEY = "warcar:data:sync:";

    private final JedisPool jedisPool;
    private final String prefixKey;
    private final int db;

    public RedisSourceController(JedisPool jedisPool) {
        this(jedisPool, DEFAULT_PREFIX_KEY, 0);
    }

    public RedisSourceController(JedisPool jedisPool, String prefixKey, int db) {
        Preconditions.checkNotNull(jedisPool);
        Preconditions.checkArgument(StringUtils.isNotBlank(prefixKey));
        this.prefixKey = prefixKey;
        this.db = db;
        this.jedisPool = jedisPool;
    }

    @Override
    public boolean set(String key, String value) {
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "key");
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "value");

        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();

            if (db > 0) {
                jedis.select(db);
            }

            return StringUtils.equalsIgnoreCase(jedis.set(prefixKey + key, value), "OK");
        } catch (Exception e) {
            LOGGER.error("set: ", e);
            return false;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public String get(String key) {
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "key");
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();

            if (db > 0) {
                jedis.select(db);
            }

            return jedis.get(prefixKey + key);
        } catch (Exception e) {
            LOGGER.error("get: ", e);
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
        return null;
    }

    @Override
    public void delete(String key) {
        if (StringUtils.isBlank(key)) {
            return;
        }
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            if (db > 0) {
                jedis.select(db);
            }
            jedis.del(prefixKey + key);
        } catch (Exception e) {
            LOGGER.error("del: ", e);
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }
}
