package com.bxm.warcar.datasync.client.cache;

import com.bxm.warcar.zk.listener.children.ChildrenChanged;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;

import java.util.Map;
import java.util.Objects;

/**
 * 数据同步 mutable 接口实现，收到数据变更后按照对象类型分发给匹配的 {@link DataSyncHandler} 实现。
 *
 * @author allen
 * @date 2025-10-22
 * @since 1.0
 * @see DataSyncHandler
 */
@Slf4j
public class DataSyncHandlerDispatcher implements MutableCache, ApplicationListener<ApplicationReadyEvent> {

    private final Map<Class<?>, DataSyncHandler> handlers = Maps.newHashMap();

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        //PathChildrenCache.startMode = NORMAL, 默认加载子节点下所有数据，并监听新的事件
        ChildrenChanged childrenChanged = event.getApplicationContext().getBean(ChildrenChanged.class);
        childrenChanged.init();

        Map<String, DataSyncHandler> beans = event.getApplicationContext().getBeansOfType(DataSyncHandler.class);
        for (Map.Entry<String, DataSyncHandler> entry : beans.entrySet()) {
            DataSyncHandler handler = entry.getValue();
            handlers.put(handler.getClazz(), handler);
            log.info("Registered {}", handler);
        }
    }

    @Override
    public Object set(String key, Object value, Class<?> clazz) throws NullPointerException {
        if (Objects.isNull(key)) {
            log.warn("key is null!");
            return null;
        }
        if (Objects.isNull(value)) {
            log.warn("value is null by key: {}", key);
            return null;
        }
        DataSyncHandler dataSyncHandler = handlers.get(clazz);
        if (Objects.isNull(dataSyncHandler)) {
            log.warn("Does not found DataSyncHandler matching class {}", clazz);
            return null;
        }
        return dataSyncHandler.set(key, value);
    }

    @Override
    public void del(String key, Class<?> clazz) {
        DataSyncHandler dataSyncHandler = handlers.get(clazz);
        if (Objects.isNull(dataSyncHandler)) {
            log.warn("Does not found DataSyncHandler matching class {}", clazz);
            return;
        }
        dataSyncHandler.del(key);
    }

    @Override
    public <T> T get(String key) throws NullPointerException {
        throw new UnsupportedOperationException();
    }

    @Override
    public Map<String, Object> getAll() {
        throw new UnsupportedOperationException();
    }

    @Override
    public long size() {
        return -1;
    }
}
