package com.bxm.warcar.dpl2.dir;

import com.bxm.warcar.utils.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 插件目录监听器
 *
 * @author allen
 * @date 2019/6/28
 * @since 1.0.0
 */
@Slf4j
public final class PluginDirMonitor implements ApplicationListener<ApplicationReadyEvent>, DisposableBean {

    private static final long DEFAULT_CHECK_TIMEOUT = 1000L;
    private final PluginChangedListener listener;
    private final String dir;
    private final long checkTimeout;
    private final boolean checkDirModified;

    private final ScheduledExecutorService scheduled = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("PluginDirMonitor"));
    private Snapshot lastSnapshot;

    public PluginDirMonitor(String dir) {
        this(new PluginChangedListener() {
        }, dir, true);
    }

    public PluginDirMonitor(PluginChangedListener listener, String dir) {
        this(listener, dir, DEFAULT_CHECK_TIMEOUT, true);
    }

    public PluginDirMonitor(PluginChangedListener listener, String dir, boolean checkDirModified) {
        this(listener, dir, DEFAULT_CHECK_TIMEOUT, checkDirModified);
    }

    public PluginDirMonitor(PluginChangedListener listener, String dir, long checkTimeout, boolean checkDirModified) {
        this.listener = listener;
        this.dir = dir;
        this.checkTimeout = checkTimeout;
        this.checkDirModified = checkDirModified;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        scheduled.scheduleWithFixedDelay(this::doCheckDir, 0, checkTimeout, TimeUnit.MILLISECONDS);
    }

    @Override
    public void destroy() throws Exception {
        scheduled.shutdownNow();
    }

    private void doCheckDir() {
        try {
            Snapshot newSnapshot = new Snapshot(dir);
            if (isModified(newSnapshot)) {
                if (log.isInfoEnabled()) {
                    log.info("checkDir: {} is changed.", dir);
                }

            Snapshot oldSnapshot = lastSnapshot;
            lastSnapshot = newSnapshot;
            listener.onChanged(new PluginChangedEvent(dir, oldSnapshot, newSnapshot));
            }
        } catch (Throwable e) {
            log.error("doCheckDir: ", e);
        }
    }

    private boolean isModified(Snapshot newSnapshot) {
        long time = System.currentTimeMillis();
        if (!checkDirModified) {
            return true;
        }
        if (null == lastSnapshot) {
            log.info("isModified 1 {}", time);
            return true;
        } else {
            if (lastSnapshot.hashCode() != newSnapshot.hashCode()) {
                log.info("isModified 2 {}", time);
                return true;
            } else {
                if (!lastSnapshot.equals(newSnapshot)) {
                    log.info("isModified 3 {}", time);
                    return true;
                }
            }
        }
        return false;
    }

    public String getDir() {
        return dir;
    }

    public long getCheckTimeout() {
        return checkTimeout;
    }
}
