package com.bxm.warcar.integration.distributed;

import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;

import java.time.Duration;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * 多次执行器
 * <p>
 * 在给定的静默时间内，确保一个操作最多被执行 n 次。这在分布式环境中非常有用，可以防止在短时间内重复处理相同的任务。
 * 例如，当多个服务实例同时收到相同的事件时，只有一个实例能够成功执行关联的操作。
 *
 * @param <T> 执行器接受的参数类型
 * @author Allen Hu
 * @since 2025-10-21
 * @see FirstTimeExecutor
 */
public class MultiTimeExecutor<T> {

    private final Counter counter;
    private final String executorName;
    private final Duration silentTime;
    private final int maxTimes;
    private final Consumer<T> executor;

    public MultiTimeExecutor(Counter counter, String executorName, Duration silentTime, int maxTimes) {
        this(counter, executorName, silentTime, maxTimes, null);
    }

    public MultiTimeExecutor(Counter counter, String executorName, Duration silentTime, int maxTimes, Consumer<T> executor) {
        this.counter = counter;
        this.executorName = executorName;
        this.silentTime = silentTime;
        this.maxTimes = maxTimes;
        this.executor = executor;
    }

    /**
     * 提交执行操作
     * <p>
     * 提交执行操作，会尝试执行操作最多 maxTimes 次。
     * 如果在静默时间内，操作执行抛出异常，会回滚操作次数。
     *
     * @param t 执行器要消费的数据
     */
    public void submit(T t) {
        execute(t, this.executor);
    }

    /**
     * 尝试执行操作
     * <p>
     * 在静默时间内，允许操作最多执行 n 次。如果操作执行抛出异常，会回滚操作次数。
     *
     * @param t        执行器要消费的数据
     * @param executor 要执行的操作
     * @see #rollback() 回滚操作
     */
    public void execute(T t, Consumer<T> executor) {
        Objects.requireNonNull(executor);
        KeyGenerator key = stringKey();
        int expireTimeInSecond = (int) silentTime.getSeconds();
        Long after = counter.incrementAndGet(key, expireTimeInSecond, value -> value <= maxTimes);
        if (after != null && after <= maxTimes) {
            try {
                executor.accept(t);
            } catch (Exception e) {
                this.rollback();
                throw e;
            }
        }
    }

    /**
     * 回滚操作
     */
    public void rollback() {
        KeyGenerator key = stringKey();
        counter.decrementAndGet(key);
    }

    private KeyGenerator stringKey() {
        return () -> KeyBuilder.build("warcar", "condition_executor", executorName);
    }
}
