package com.bxm.warcar.integration.distributed.zk;

import com.bxm.warcar.integration.distributed.DistributedLock;
import com.bxm.warcar.integration.distributed.DistributedLockBeanFactory;
import com.bxm.warcar.integration.distributed.DistributedLockBus;
import com.bxm.warcar.integration.distributed.lock.ZooKeeperDistributedLock;
import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.util.ClassUtils;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * @author allen
 * @since 1.0.0
 */
public class ZooKeeperDistributedLockBus implements DistributedLockBus {

    private final static Logger logger = LoggerFactory.getLogger(ZooKeeperDistributedLockBus.class);

    private final ConcurrentMap<String, ZooKeeperDistributedLock> locks = new ConcurrentHashMap<>(64);

    private final ZkClientHolder zkClientHolder;

    public ZooKeeperDistributedLockBus(DistributedLockBeanFactory distributedLockBeanFactory, ZkClientHolder zkClientHolder) {
        Preconditions.checkNotNull(zkClientHolder);
        this.zkClientHolder = zkClientHolder;

        final Map<Method, Object> mapping = distributedLockBeanFactory.getMapping();
        for (Map.Entry<Method, Object> entry : mapping.entrySet()) {
            this.processScheduled(entry.getKey(), entry.getValue());
        }
    }

    private void processScheduled(Method method, Object bean) {
        String className = getClassName(bean);
        String path = ZooKeeperDistributedPathUtils.createPath(className, method);
        ZooKeeperDistributedLock lock = new ZooKeeperDistributedLock(zkClientHolder, path);
        if (locks.containsKey(path)) {
            throw new RuntimeException("Distributed lock '" + path + "' for ZooKeeper already exist!");
        }
        this.locks.putIfAbsent(path, lock);

        logger.info("{} created distributed lock on bean {}", method, bean);
    }

    @Override
    public DistributedLock getLock(Object bean, Method method) {
        String path = ZooKeeperDistributedPathUtils.createPath(getClassName(bean), method);
        return this.locks.get(path);
    }

    private String getClassName(Object bean) {
        Class<?> targetClass = AopUtils.getTargetClass(bean);
        String className = targetClass.getName();
        if (ClassUtils.isCglibProxy(bean)) {
            className = StringUtils.split(className, ClassUtils.CGLIB_CLASS_SEPARATOR)[0];
        }
        return className;
    }
}
