package com.bxm.warcar.integration.distributed.delayed;

import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.utils.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RBlockingQueue;
import org.redisson.api.RDelayedQueue;
import org.redisson.api.RedissonClient;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

/**
 * 分布式延时任务队列工具类
 *
 * @author Allen Hu
 * @date 2025/9/5
 * @see LatestOnlyDelayedTaskQueueTool
 */
@Slf4j
public class DelayedTaskQueueTool<V> extends LifeCycle {

    private final RBlockingQueue<V> blockingQueue;
    private final RDelayedQueue<V> delayedQueue;
    private final ThreadPoolExecutor consumerThread;
    private final Consumer<V> consumer;
    private final String taskName;
    private final AtomicBoolean stop = new AtomicBoolean(false);

    /**
     * @param redissonClient redisson客户端
     * @param taskName       任务名称，要求唯一性
     * @param consumer       任务消费者，当延迟时间到达时被调用
     */
    public DelayedTaskQueueTool(RedissonClient redissonClient, String taskName, Consumer<V> consumer) {
        this.taskName = taskName;
        this.blockingQueue = redissonClient.getBlockingQueue(taskName);
        this.delayedQueue = redissonClient.getDelayedQueue(blockingQueue);
        this.consumerThread = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(1), new NamedThreadFactory(String.format("delayed-%s-queue-tool", taskName)));
        this.consumer = consumer;
    }

    /**
     * 插入任务，
     * 注意：如果队列中存在相同值的任务，也会新增延迟任务
     *
     * @param value  任务值
     * @param delay  延迟时间
     * @param timeUnit 时间单位
     */
    public void offer(V value, long delay, TimeUnit timeUnit) {
        delayedQueue.offer(value, delay, timeUnit);
    }

    /**
     * 移除指定值的任务，该方法对性能有影响，谨慎使用
     * 注意：如果队列中存在多个相同值的任务，只会移除队列中的其中一个任务。
     *
     * @param value 任务值
     * @return 是否移除成功
     * @see #removeAll(Object)
     */
    public boolean remove(V value) {
        return delayedQueue.remove(value);
    }

    /**
     * 移除指定值的任务，该方法对性能有影响，谨慎使用
     * 注意：如果队列中存在多个相同值的任务，会移除所有相同值的任务
     *
     * @param value 任务值
     */
    public void removeAll(V value) {
        boolean removed;
        do {
            removed = remove(value);
        } while (removed);
    }

    /**
     * 清空延迟队列
     */
    public void clear() {
        delayedQueue.clear();
    }

    public int size() {
        return delayedQueue.size();
    }

    @Override
    protected void doInit() {
        consumerThread.execute(() -> {
            while (true) {
                try {
                    if (this.stop.get()) {
                        break;
                    }
                    log.info("delayed {} queue tool start consume", taskName);
                    V value = blockingQueue.take();
                    log.info("delayed {} queue tool consume value: {}", taskName, value);
                    consumer.accept(value);
                } catch (InterruptedException ignored) {
                } catch (Exception e) {
                    log.error("delayed {} queue tool consume error", taskName, e);
                }
            }
        });
        log.info("delayed {} queue tool init success", taskName);
    }

    @Override
    protected void doDestroy() {
        this.stop.compareAndSet(false, true);
        consumerThread.shutdownNow();
        log.info("delayed {} queue tool shutdown success", taskName);
    }

}
