package com.bxm.warcar.integration.distributed.leader;

import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.leader.LeaderSelector;
import org.apache.curator.framework.recipes.leader.LeaderSelectorListenerAdapter;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.io.Closeable;
import java.util.List;

/**
 * <p>LeaderSelectorExecutorBeanFactory</p>
 * <p>
 * 用于自动为实现 LeaderSelectorExecutor 接口的 Spring Bean 创建并管理 LeaderSelector 实例，实现分布式 Leader 选举。
 * 在 Bean 初始化后，自动注册 LeaderSelector，并在成为 Leader 时调用对应 Bean 的 execute 方法。
 * 实现 Closeable 接口，容器销毁时自动关闭所有 LeaderSelector，释放资源。
 * </p>
 * 主要功能：<br/>
 * 1. 自动发现并处理 LeaderSelectorExecutor 类型的 Bean。<br/>
 * 2. 为每个 Bean 创建 LeaderSelector，并监听 Leader 状态变化。<br/>
 * 3. 成为 Leader 时执行 Bean 的业务逻辑。<br/>
 * 4. 统一管理 LeaderSelector 生命周期，支持自动关闭。<br/>
 *
 * @author Allen Hu
 * @date 2025/10/20
 */
@Slf4j
public class LeaderSelectorExecutorBeanFactory implements BeanPostProcessor, Closeable {

    private final ZkClientHolder zkClientHolder;
    private final List<LeaderSelector> leaderSelectorList = Lists.newArrayList();

    public LeaderSelectorExecutorBeanFactory(ZkClientHolder zkClientHolder) {
        this.zkClientHolder = zkClientHolder;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.getTargetClass(bean);

        boolean assignableFrom = LeaderSelectorExecutor.class.isAssignableFrom(targetClass);
        if (assignableFrom) {
            LeaderSelectorExecutor leaderSelectorExecutor = (LeaderSelectorExecutor) bean;
            this.createLeaderSelectorAndStart(leaderSelectorExecutor);
        }

        return bean;
    }

    @Override
    public void close() {
        leaderSelectorList.forEach(LeaderSelector::close);
    }

    private void createLeaderSelectorAndStart(LeaderSelectorExecutor leaderSelectorExecutor) {
        String className = leaderSelectorExecutor.getClass().getName();
        LeaderSelector leaderSelector = new LeaderSelector(zkClientHolder.get(), "/warcar/leader-selector/" + className, new LeaderSelectorListenerAdapter() {
            @Override
            public void takeLeadership(CuratorFramework client) {
                try {
                    leaderSelectorExecutor.execute();
                } catch (Exception e) {
                    log.error("LeaderSelectorExecutor execute error", e);
                }
            }
        });

        leaderSelectorList.add(leaderSelector);
        try {
            leaderSelector.autoRequeue();
            leaderSelector.start();
            log.info("LeaderSelector {} start success", className);
        } catch (Exception e) {
            log.error("LeaderSelector {} start error", className, e);
        }
    }

}
