package com.bxm.warcar.integration.distributed.scheduler;

import com.bxm.warcar.integration.distributed.DistributedLock;
import com.bxm.warcar.integration.distributed.FirstTimeExecutor;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.function.Consumer;

/**
 * @author Allen Hu
 * @date 2025/10/22
 */
@Slf4j
public class ThreadPoolTaskDistributedScheduler implements DistributedScheduler {

    private final ThreadPoolTaskScheduler threadPoolTaskScheduler = new ThreadPoolTaskScheduler();
    private final Map<String, ScheduledFuture<?>> scheduledFutures = new ConcurrentHashMap<>();

    public ThreadPoolTaskDistributedScheduler(int poolSize) {
        threadPoolTaskScheduler.setPoolSize(poolSize);
        threadPoolTaskScheduler.initialize();
    }

    /**
     * 调度指定的调度器
     *
     * @param scheduler 调度器
     */
    @Override
    public void schedule(Scheduler scheduler) {
        String name = scheduler.name();
        Runnable taskRunnable = scheduler.taskRunnable();
        Preconditions.checkArgument(StringUtils.isNotBlank(name), "name must not be blank");
        Preconditions.checkNotNull(taskRunnable, "taskRunnable must not be null");
        if (scheduledFutures.containsKey(name)) {
            log.warn("scheduler {} already scheduled", name);
            return;
        }

        ScheduledFuture<?> future = threadPoolTaskScheduler.schedule(() -> {
            if (scheduler instanceof LockScheduler) {
                scheduleLockScheduler((LockScheduler) scheduler);
            } else if (scheduler instanceof AtomicOnceScheduler) {
                scheduleAtomicOnceScheduler((AtomicOnceScheduler) scheduler);
            } else {
                scheduleScheduler(scheduler);
            }
        }, triggerContext -> {
            CronTrigger cronTrigger = new CronTrigger(scheduler.cron());
            return cronTrigger.nextExecutionTime(triggerContext);
        });
        scheduledFutures.put(name, future);
        log.info("scheduler {} scheduled", name);
    }

    /**
     * 取消指定名称的调度器，如果调度器不存在，则忽略
     *
     * @param name 调度器名称
     */
    @Override
    public void cancel(String name) {
        Preconditions.checkArgument(StringUtils.isNotBlank(name), "name must not be blank");
        ScheduledFuture<?> future = scheduledFutures.remove(name);
        if (null != future) {
            future.cancel(true);
            log.info("scheduler {} cancelled", name);
        }
    }

    private void scheduleScheduler(Scheduler scheduler) {
        Runnable taskRunnable = scheduler.taskRunnable();
        Class<?> targetClass = AopUtils.getTargetClass(scheduler);
        try {
            log.debug("scheduled task {} acquired lock and run", targetClass.getName());
            taskRunnable.run();
        } catch (Exception e) {
            log.error("scheduled task {} failed", targetClass.getName(), e);
        }
    }

    private void scheduleLockScheduler(LockScheduler lockScheduler) {
        Runnable taskRunnable = lockScheduler.taskRunnable();
        DistributedLock distributedLock = lockScheduler.distributedLock();
        long idleTime = lockScheduler.idleTime();
        Class<?> targetClass = AopUtils.getTargetClass(lockScheduler);
        boolean acquired = false;
        try {
            acquired = distributedLock.acquire();
            if (acquired) {
                log.debug("scheduled task {} acquired lock and run", targetClass.getName());
                taskRunnable.run();
            }
        } catch (Exception e) {
            log.error("scheduled task {} failed", targetClass.getName(), e);
        } finally {
            try {
                Thread.sleep(idleTime);
            } catch (InterruptedException e) {
                log.error("scheduled task {} sleep interrupted", targetClass.getName(), e);
            }
            if (acquired) {
                distributedLock.release();
            }
        }
    }

    private void scheduleAtomicOnceScheduler(AtomicOnceScheduler atomicOnceScheduler) {
        Runnable taskRunnable = atomicOnceScheduler.taskRunnable();
        FirstTimeExecutor<Object> firstTimeExecutor = atomicOnceScheduler.firstTimeExecutor();
        Class<?> targetClass = AopUtils.getTargetClass(atomicOnceScheduler);
        try {
            firstTimeExecutor.execute(new Object(), new Consumer<Object>() {
                @Override
                public void accept(Object o) {
                    log.debug("scheduled task {} first time run", targetClass.getName());
                    taskRunnable.run();
                }
            });
        } catch (Exception e) {
            log.error("scheduled task {} failed", targetClass.getName(), e);
        }
    }

    @Override
    public void close() throws IOException {
        threadPoolTaskScheduler.shutdown();
    }
}
