/*
 * Copyright 2016 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * textile.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it only
 * in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */

package com.bxm.warcar.ip.impl;

import com.bxm.warcar.ip.IP;
import com.bxm.warcar.ip.IpLibrary;
import com.bxm.warcar.utils.LifeCycle;
import com.google.common.base.Preconditions;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

/**
 * <h3>ipip.net</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/14
 */
public class IpIpNetIpLibrary extends LifeCycle implements IpLibrary {

    private static final Logger LOGGER = LoggerFactory.getLogger(IpIpNetIpLibrary.class);

    private final File ipFile;
    private final InputStream inputStream;
    public boolean enableFileWatch = false;

    public IpIpNetIpLibrary(File ipFile) {
        Preconditions.checkNotNull(ipFile);
        this.ipFile = ipFile;
        this.inputStream = null;
    }

    public IpIpNetIpLibrary(InputStream inputStream) {
        this.ipFile = null;
        this.inputStream = inputStream;
    }

    @Override
    public IP find(String ip) {
        String[] array = findIp(ip);
        if (ArrayUtils.isEmpty(array))
            return null;

        if (array.length > 11) {
            return new IP(array[0],
                    array[1],
                    array[2],
                    array[3],
                    array[4],
                    array[5],
                    array[6],
                    array[7],
                    array[8],
                    array[9],
                    NumberUtils.toInt(array[10], -1),
                    array[11]);
        }

        if (array.length > 2) {
            return new IP(array[0],
                    array[1],
                    array[2]);
        }

        return null;
    }

    @Override
    protected void doInit() {
        long start = System.currentTimeMillis();
        this.load();
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Ip library load finished in {} ms", (System.currentTimeMillis() - start));
        }
        if (enableFileWatch) {
            watch();
        }
    }

    @Override
    protected void doDestroy() {

    }

    public void setEnableFileWatch(boolean enableFileWatch) {
        this.enableFileWatch = enableFileWatch;
    }

    private int offset;
    private int[] index = new int[65536];
    private ByteBuffer dataBuffer;
    private ByteBuffer indexBuffer;
    private Long lastModifyTime = 0L;
    private ReentrantLock lock = new ReentrantLock();

    private String[] findIp(String ip) {
        if (null == dataBuffer) {
            throw new NullPointerException("no init.");
        }

        String[] ips = ip.split("\\.");
        int prefix_value = (Integer.valueOf(ips[0]) * 256 + Integer.valueOf(ips[1]));
        long ip2long_value = ip2long(ip);
        int start = index[prefix_value];
        int max_comp_len = offset - 262144 - 4;
        long tmpInt;
        long index_offset = -1;
        int index_length = -1;
        byte b = 0;
        for (start = start * 9 + 262144; start < max_comp_len; start += 9) {
            tmpInt = int2long(indexBuffer.getInt(start));
            if (tmpInt >= ip2long_value) {
                index_offset = bytesToLong(b, indexBuffer.get(start + 6), indexBuffer.get(start + 5), indexBuffer.get(start + 4));
                index_length = ((0xFF & indexBuffer.get(start + 7)) << 8) + (0xFF & indexBuffer.get(start + 8));
                break;
            }
        }

        byte[] areaBytes;

        lock.lock();
        try {
            dataBuffer.position(offset + (int) index_offset - 262144);
            areaBytes = new byte[index_length];
            dataBuffer.get(areaBytes, 0, index_length);
        } finally {
            lock.unlock();
        }

        return new String(areaBytes, Charset.forName("UTF-8")).split("\t", -1);
    }

    private void watch() {
        if (null != ipFile && enableFileWatch) {
            Executors.newScheduledThreadPool(1).scheduleAtFixedRate(new Runnable() {
                @Override
                public void run() {
                    long time = ipFile.lastModified();
                    if (time > lastModifyTime) {
                        load();
                    }
                }
            }, 1000L, 5000L, TimeUnit.MILLISECONDS);
        }
    }

    private void load() {
        lastModifyTime = null != ipFile ? ipFile.lastModified() : null;
        lock.lock();
        try {
            byte[] bytes =
                    null == ipFile ? getBytesByInputStream(inputStream) : getBytesByFile(ipFile);
            dataBuffer = ByteBuffer.wrap(bytes);
            dataBuffer.position(0);
            offset = dataBuffer.getInt(); // indexLength
            byte[] indexBytes = new byte[offset];
            dataBuffer.get(indexBytes, 0, offset - 4);
            indexBuffer = ByteBuffer.wrap(indexBytes);
            indexBuffer.order(ByteOrder.LITTLE_ENDIAN);

            for (int i = 0; i < 256; i++) {
                for (int j = 0; j < 256; j++) {
                    index[i * 256 + j] = indexBuffer.getInt();
                }
            }
            indexBuffer.order(ByteOrder.BIG_ENDIAN);
        } catch (IOException e) {
            throw new RuntimeException("load", e);
        } finally {
            lock.unlock();
        }
    }

    private byte[] getBytesByFile(File ipFile) throws IOException {
        InputStream fin = null;

        long len = ipFile.length();

        byte[] bs = new byte[new Long(len).intValue()];
        try {
            fin = new FileInputStream(ipFile);
            int readBytesLength = 0;
            int i;
            while ((i = fin.available()) > 0) {
                fin.read(bs, readBytesLength, i);
                readBytesLength += i;
            }
        } catch (IOException ioe) {
            ioe.printStackTrace();
        } finally {
            try {
                if (fin != null) {
                    fin.close();
                }
            } catch (IOException e){
                e.printStackTrace();
            }
        }

        return bs;
    }

    private byte[] getBytesByInputStream(InputStream inputStream) throws IOException {
        byte[] buffer = new byte[inputStream.available()];
        IOUtils.readFully(inputStream, buffer);
        return buffer;
    }

    private long bytesToLong(byte a, byte b, byte c, byte d) {
        return int2long((((a & 0xff) << 24) | ((b & 0xff) << 16) | ((c & 0xff) << 8) | (d & 0xff)));
    }
    private int str2Ip(String ip)  {
        String[] ss = ip.split("\\.");
        int a, b, c, d;
        a = Integer.parseInt(ss[0]);
        b = Integer.parseInt(ss[1]);
        c = Integer.parseInt(ss[2]);
        d = Integer.parseInt(ss[3]);
        return (a << 24) | (b << 16) | (c << 8) | d;
    }

    private long ip2long(String ip)  {
        return int2long(str2Ip(ip));
    }

    private long int2long(int i) {
        long l = i & 0x7fffffffL;
        if (i < 0) {
            l |= 0x080000000L;
        }
        return l;
    }

    public static void main(String[] args) {
        IpIpNetIpLibrary library = new IpIpNetIpLibrary(new File("E:\\IP.datx"));
        library.init();
        System.out.println(library.find("117.136.79.102"));
        System.out.println(library.find("112.17.87.139"));
        System.out.println(library.find("73.32.192.246"));
        System.out.println(library.find("220.176.34.17"));
        System.out.println(library.find("220.191.32.0"));
    }
}
