package com.bxm.warcar.logging.extension;

import com.aliyun.openservices.aliyun.log.producer.*;
import com.aliyun.openservices.aliyun.log.producer.errors.ProducerException;
import com.aliyun.openservices.log.common.LogItem;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * @author allen
 * @date 2022-05-30
 * @since 1.0
 */
@Slf4j
public class AliyunLogServiceCustomLoggingWriter implements CustomLoggingWriter, InitializingBean, DisposableBean {

    private final ProducerConfig producerConfig;
    private final ProjectConfig projectConfig;
    private final String logStore;

    private ThreadPoolExecutor executor;

    private LogProducer producer;

    public AliyunLogServiceCustomLoggingWriter(ProducerConfig producerConfig, ProjectConfig projectConfig, String logStore) {
        this.producerConfig = producerConfig;
        this.projectConfig = projectConfig;
        this.logStore = logStore;
    }

    @Override
    public void write(Map<String, String> data) {
        if (Objects.nonNull(executor)) {
            executor.submit(new Runnable() {
                @Override
                public void run() {
                    send(data);
                }
            });
        } else {
            send(data);
        }
    }

    private void send(Map<String, String> data) {
        if (MapUtils.isEmpty(data)) {
            return;
        }

        LogItem logItem = new LogItem();

        for (Map.Entry<String, String> entry : data.entrySet()) {
            logItem.PushBack(entry.getKey(), entry.getValue());
        }

        try {
            producer.send(projectConfig.getProject(), logStore, logItem, new Callback() {
                @Override
                public void onCompletion(Result result) {
                    if (!result.isSuccessful()) {
                        log.warn("{}", result.getErrorMessage());
                    }
                }
            });
        } catch (Exception e) {
            log.error("send: ", e);
        }
    }

    @Override
    public void destroy() throws Exception {
        try {
            this.producer.close();
        } catch (InterruptedException | ProducerException e) {
            log.error("close: ", e);
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.producer = new LogProducer(producerConfig);
        this.producer.putProjectConfig(projectConfig);
    }

    public void setExecutor(ThreadPoolExecutor executor) {
        this.executor = executor;
    }
}
