package com.bxm.warcar.logging.support;

import com.aliyun.openservices.aliyun.log.producer.*;
import com.aliyun.openservices.aliyun.log.producer.errors.ProducerException;
import com.aliyun.openservices.log.common.LogItem;
import com.bxm.warcar.logging.Logging;
import com.bxm.warcar.logging.LoggingWriter;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.TypeHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 阿里云日志服务实现
 *
 * https://sls.console.aliyun.com/
 * https://github.com/aliyun/aliyun-log-java-producer
 *
 * @author allen
 * @date 2021-07-30
 * @since 1.0
 */
@Slf4j
public class AliyunLogServiceLoggingWriter implements LoggingWriter, InitializingBean, DisposableBean {

    private final ProducerConfig producerConfig;
    private final ProjectConfig projectConfig;
    private final String logStore;

    private ThreadPoolExecutor executor;
    private LoggingWriter fallback;

    private LogProducer producer;

    public AliyunLogServiceLoggingWriter(ProducerConfig producerConfig, ProjectConfig projectConfig, String logStore) {
        this.producerConfig = producerConfig;
        this.projectConfig = projectConfig;
        this.logStore = logStore;
    }

    @Override
    public void write(Logging logging) {
        if (Objects.nonNull(executor)) {
            executor.submit(new Runnable() {
                @Override
                public void run() {
                    send(logging);
                }
            });
        } else {
            send(logging);
        }
    }

    private void send(Logging logging) {
        String topic = logging.getApplicationId();

        LogItem logItem = new LogItem();

        logItem.PushBack("applicationId", logging.getApplicationId());
        logItem.PushBack("requestedSessionId", logging.getRequestedSessionId());
        logItem.PushBack("scheme", logging.getScheme());
        logItem.PushBack("host", logging.getHost());
        logItem.PushBack("port", TypeHelper.castToString(logging.getPort()));
        logItem.PushBack("method", logging.getMethod());
        logItem.PushBack("path", logging.getPath());
        logItem.PushBack("queryString", logging.getQueryString());
        logItem.PushBack("params", JsonHelper.convert(logging.getParams()));
        logItem.PushBack("clientIp", logging.getClientIp());
        logItem.PushBack("serverIp", logging.getServerIp());
        logItem.PushBack("headers", JsonHelper.convert(logging.getHeaders()));
        logItem.PushBack("startTime", TypeHelper.castToString(logging.getStartTime()));
        logItem.PushBack("endTime", TypeHelper.castToString(logging.getEndTime()));
        logItem.PushBack("requestBody", logging.getRequestBody());
        logItem.PushBack("responseBody", logging.getResponseBody());
        logItem.PushBack("consumeTimeInMillis", TypeHelper.castToString(logging.getConsumeTimeInMillis()));
        logItem.PushBack("success", TypeHelper.castToString(logging.isSuccess()));
        logItem.PushBack("throwable", logging.getThrowable());
        logItem.PushBack("operator", logging.getOperator());

        try {
            producer.send(projectConfig.getProject(), logStore, topic, null, logItem, new Callback() {
                @Override
                public void onCompletion(Result result) {
                    if (log.isDebugEnabled()) {
                        log.debug("send: {}", result);
                    }
                    if (!result.isSuccessful()) {
                        if (Objects.nonNull(fallback)) {
                            fallback.write(logging);
                        }
                    }
                }
            });
        } catch (InterruptedException | ProducerException e) {
            log.error("send: ", e);
        }
    }

    @Override
    public void destroy() throws Exception {
        try {
            this.producer.close();
        } catch (InterruptedException | ProducerException e) {
            log.error("close: ", e);
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.producer = new LogProducer(producerConfig);
        this.producer.putProjectConfig(projectConfig);
    }

    public void setFallback(LoggingWriter fallback) {
        this.fallback = fallback;
    }

    public void setExecutor(ThreadPoolExecutor executor) {
        this.executor = executor;
    }
}
