package com.bxm.warcar.metrics;

import com.google.common.util.concurrent.AtomicDouble;
import org.springframework.boot.actuate.metrics.GaugeService;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * @author allen
 * @since 1.0.0
 */
public class DefaultConsumeTimeMetric implements ConsumeTimeMetric {

    private final GaugeService gaugeService;

    public DefaultConsumeTimeMetric(GaugeService gaugeService) {
        this.gaugeService = gaugeService;
    }

    @Override
    public void submit(String metricName, double value) {
        increaseTask(metricName, value);
        gaugeService.submit(metricName, value);
    }

    private final ConcurrentMap<String, AtomicDouble> lastTaskConsumeTimeTable = new ConcurrentHashMap<String, AtomicDouble>();

    private final ConcurrentMap<String, AtomicDouble> largestTaskConsumeTimeTable = new ConcurrentHashMap<String, AtomicDouble>();

    private void increaseTask(String serviceName, double consumingTime) {
        AtomicDouble lastTaskConsumeTime = lastTaskConsumeTimeTable.putIfAbsent(serviceName,
                new AtomicDouble(0L));
        if (null == lastTaskConsumeTime) {
            lastTaskConsumeTime = lastTaskConsumeTimeTable.get(serviceName);
        }
        lastTaskConsumeTime.set(consumingTime);

        AtomicDouble largestTimeContainer = largestTaskConsumeTimeTable.putIfAbsent(
                serviceName, new AtomicDouble(0));
        if (null == largestTimeContainer) {
            largestTimeContainer = largestTaskConsumeTimeTable.get(serviceName);
        }

        processLargestTaskConsumeTime(largestTimeContainer, consumingTime);
    }

    private void processLargestTaskConsumeTime(AtomicDouble largestTimeContainer, double consumingTime) {
        int retry = 0;
        boolean done = false;
        do {
            long expect = largestTimeContainer.longValue();
            if (consumingTime > expect) {
                done = largestTimeContainer.compareAndSet(expect, consumingTime);
            }
            retry++;
        } while ((!done) && (retry < 3));
    }
}
