package com.bxm.warcar.micrometer;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.binder.MeterBinder;
import lombok.Setter;
import org.aspectj.lang.ProceedingJoinPoint;

/**
 * 简单的指标统计切面工具类，用于统计服务方法的执行次数和耗时
 *
 * <pre>
 *     -- e.g. --
 *     &#64;Aspect
 *     public class TableHandlerMeterAspect extends SimpleMicrometerAspectTool {
 *
 *         &#64;Around("execution(* com.bxm.lovelink.cm.data.TableHandler+.on*(..))")
 *         public Object timeRecommendService(ProceedingJoinPoint pjp) throws Throwable {
 *             return super.doMeter(pjp);
 *         }
 *     }
 * </pre>
 *
 * @author Allen Hu
 * @date 2025-11-05
 */
public class SimpleMicrometerAspectTool implements MeterBinder {

    private MeterRegistry meterRegistry;
    @Setter
    private boolean timerEnabled = true;
    @Setter
    private String prefixName = "service.execution";

    @Override
    public void bindTo(MeterRegistry registry) {
        this.meterRegistry = registry;
    }

    /**
     * 统计服务方法的执行次数和耗时
     *
     * @param pjp 连接点
     * @return 方法执行结果
     * @throws Throwable 方法执行异常
     */
    public Object doMeter(ProceedingJoinPoint pjp) throws Throwable {
        if (null != meterRegistry) {
            String className = pjp.getTarget().getClass().getSimpleName();
            String methodName = pjp.getSignature().getName();
            Timer.Sample sample = null;
            if (timerEnabled) {
                sample = Timer.start(meterRegistry);
            }
            try {
                Counter.builder(prefixName + ".count")
                        .description("Count of service methods execution")
                        .tag("service", className)
                        .tag("method", methodName)
                        .register(meterRegistry)
                        .increment();
                return pjp.proceed();
            } catch (Throwable e) {
                Counter.builder(prefixName + ".count.error")
                        .description("Count of service methods execution error")
                        .tag("service", className)
                        .tag("method", methodName)
                        .register(meterRegistry)
                        .increment();
                throw e;
            } finally {
                if (timerEnabled && null != sample) {
                    sample.stop(Timer.builder(prefixName + ".time")
                            .description("Time spent on service methods")
                            .tag("service", className)
                            .tag("method", methodName)
                            .register(meterRegistry));
                }
            }
        } else {
            return pjp.proceed();
        }
    }
}
