package com.bxm.warcar.mq.rocketmq;


import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.Producer;
import com.bxm.warcar.mq.SendException;
import com.bxm.warcar.mq.SendResult;
import com.bxm.warcar.utils.LifeCycle;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.client.producer.SendStatus;

import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 基于RocketMQ的实现。
 * 参考资料：<a href="https://rocketmq.apache.org">https://rocketmq.apache.org</a>
 * @author allen
 */
public class RocketmqProducer extends LifeCycle implements Producer {
    
    private final DefaultMQProducer producer;
    private final String nameSrvAddr;
    private final AtomicBoolean started = new AtomicBoolean(false);

    public RocketmqProducer(String nameSrvAddr) {
        this(nameSrvAddr, Producer.DEFAULT_PRODUCER_GROUP);
    }

    public RocketmqProducer(String nameSrvAddr, String producerGroup) {
        this.producer = new DefaultMQProducer(producerGroup);
        this.nameSrvAddr = nameSrvAddr;
        this.producer.setNamesrvAddr(nameSrvAddr);
    }

    @Override
    public SendResult send(Message message) throws SendException {
        try {
            org.apache.rocketmq.common.message.Message msg = new org.apache.rocketmq.common.message.Message();
            msg.setTopic(message.getTopic());
            msg.setBody(message.getBody());
            msg.setTags(message.getTags());
            int level = message.getDelayTimeLevel();
            if (level > 0) {
                msg.setDelayTimeLevel(level);
            }
            org.apache.rocketmq.client.producer.SendResult sendResult = producer.send(msg);
            if (null == sendResult) {
                throw new SendException();
            }
            if (sendResult.getSendStatus() != SendStatus.SEND_OK) {
                throw new SendException(String.format("Send failed! %s", sendResult.getSendStatus()));
            }
            
            return new SendResult(sendResult.getMsgId());
        }
        catch (Exception e) {
            throw new SendException(e);
        }
    }

    @Override
    protected void doInit() {
        try {
            producer.start();
            started.set(true);
        }
        catch (MQClientException e) {
            throw new RuntimeException("doInit:", e);
        }
    }

    @Override
    protected void doDestroy() {
        producer.shutdown();
        started.set(false);
    }

    @Override
    public void start() {
        init();
    }

    @Override
    public void close() {
        destroy();
    }

    @Override
    public boolean isStarted() {
        return null != producer && started.get();
    }

    public DefaultMQProducer getProducer() {
        return producer;
    }

    public String getNameSrvAddr() {
        return nameSrvAddr;
    }
}
