package com.bxm.warcar.mq.redis;

import com.bxm.warcar.mq.*;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * <h3>集群模式接收消息</h3>
 * <p>
 *     发布消息 {@link JedisConsumer}
 *     注意此方法不支持事务, 广播消息可选择{@link JedisSubscriber}
 * </p>
 * @author allen
 * @date 2019/10/21
 * @since 1.0.0
 */
public class JedisConsumer extends LifeCycle implements Consumer {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisConsumer.class);
    private final ThreadPoolExecutor consumer = new ThreadPoolExecutor(1, 1, 0, TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("JedisConsumer"));
    private final JedisPool jedisPool;
    private final SingleMessageListener messageListener;
    private final AtomicBoolean shutdown = new AtomicBoolean(false);

    public JedisConsumer(JedisPool jedisPool, SingleMessageListener messageListener) {
        this.jedisPool = jedisPool;
        this.messageListener = messageListener;
    }


    @Override
    protected void doInit() {
        String topic = messageListener.getTopic();
        consumer.execute(() -> {
            for (;;) {
                if (shutdown.get()) {
                    break;
                }
                if (null == jedisPool || jedisPool.isClosed()) {
                    threadDelay(1000);
                    continue;
                }
                Jedis jedis = null;
                jedis = jedisPool.getResource();
                try {
                    List<String> list = jedis.blpop(1, RedisConst.key(topic));
                    if (CollectionUtils.isEmpty(list)) {
                        continue;
                    }
                    for (int index = 0; index < list.size(); index++) {
                        if (index == 0) {
                            // The first value is key name.
                            continue;
                        }
                        String json = list.get(index);
                        Message message = JsonHelper.convert(json, Message.class);
                        if (null != message) {
                            ConsumeStatus status = messageListener.consume(message, jedis);
                            if (status == ConsumeStatus.RECONSUME_LATER) {
                                // 如果消费失败，重新放回队列，等待下一次消费。
                                jedis.rpush(RedisConst.key(topic), json);
                                // 加延迟的作用是防止一直消费失败，CPU被一直占用。
                                threadDelay(500);
                            }
                        }
                    }
                } catch (Exception e) {
                    LOGGER.error("", e);
                } finally {
                    if (null != jedis) {
                        jedis.close();
                    }
                }
            }
        });
    }

    private void threadDelay(long millis) {
        try {
            Thread.sleep(millis);
        } catch (InterruptedException ignore) {
        }
    }

    @Override
    protected void doDestroy() {
        consumer.shutdown();
        shutdown.compareAndSet(false, true);
        try {
            consumer.awaitTermination(5, TimeUnit.SECONDS);
        } catch (InterruptedException ignore) {
        }
    }

    @Override
    public void suspend() {

    }

    @Override
    public void shutdown() {
        this.doDestroy();
    }

    @Override
    public void start() {
        this.doInit();
    }

    @Override
    public boolean isStarted() {
        return false;
    }

    @Override
    public Listener getMessageListener() {
        return messageListener;
    }
}
