package com.bxm.warcar.utils.http.pool;

import com.bxm.warcar.utils.http.HttpPool;
import com.bxm.warcar.utils.http.RestTemplateUtils;
import com.bxm.warcar.utils.http.properties.HttpProperties;
import com.google.common.collect.Maps;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.Map;

/**
 * <h3>类的基本描述</h3>
 * <p> 单个请求 {@link RestTemplateUtils}</p>
 * @author hcmony
 * @since V1.0.0, 2019/6/20 15:32
 */
public class RestTemplatePool implements HttpPool {

    private final RestTemplate restTemplate;
    private final HttpProperties httpProperties;

    private ClientHttpRequestFactory httpRequestFactory() {
        return new HttpComponentsClientHttpRequestFactory(httpClient());
    }

    public RestTemplatePool(HttpProperties httpProperties) {
        this.httpProperties = httpProperties;
        this.restTemplate = restTemplate();
    }

    private RestTemplate restTemplate() {
        return new RestTemplate(httpRequestFactory());
    }

    private HttpClient httpClient() {
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", SSLConnectionSocketFactory.getSocketFactory())
                .build();
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(registry);
        connectionManager.setMaxTotal(httpProperties.getMaxTotal());
        connectionManager.setDefaultMaxPerRoute(httpProperties.getDefaultMaxPerRoute());

        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(httpProperties.getSocketTimeout())
                .setConnectTimeout(httpProperties.getConnectTimeout())
                .setConnectionRequestTimeout(httpProperties.getConnectionRequestTimeout())
                .build();

        return HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager)
                .build();
    }


    public RestTemplate builder() {
        return this.restTemplate;
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url http://api.bianxiao.com?test=1
     * @return
     */
    @Override
    public String get(String url) {
        return exchange(url, HttpMethod.GET).getBody();
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url     http://api.bianxiao.com?test=1
     * @param headers heders传参
     * @return
     */
    @Override
    public String get(String url, Map<String, String> headers) {
        return get(url, Maps.newHashMap(), headers);
    }

    /**
     * get 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String get(String url, Map<String, Object> params, Map<String, String> headers) {
        return exchange(url, headers, params, HttpMethod.GET, String.class).getBody();
    }

    /**
     * post 请求
     *
     * @param url http://api.bianxiao.com
     * @return
     */
    @Override
    public String post(String url) {
        return exchange(url, HttpMethod.POST).getBody();
    }

    /**
     * post 请求
     *
     * @param url    http://api.bianxiao.com
     * @param params params传参
     * @return
     */
    @Override
    public String post(String url, Map<String, Object> params) {
        return post(url, params, Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String post(String url, Map<String, Object> params, Map<String, String> headers) {
        return exchange(url, headers, params, HttpMethod.POST, String.class).getBody();
    }

    /**
     * post 请求 支持@RequestBody 传参请求
     *
     * @param url     http://api.bianxiao.com
     * @param o       实体传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String postRequestBody(String url, Object o, Map<String, String> headers) {
        return exchange(url, headers, o, HttpMethod.POST, String.class).getBody();
    }

    /**
     * @param url    http://api.bianxiao.com
     * @param method HttpMethod.POST,HttpMethod.GET {@link HttpMethod}
     * @return
     */
    public HttpEntity<String> exchange(String url, HttpMethod method) {
        return exchange(url, method, Maps.newHashMap());
    }

    /**
     * @param url          http://api.bianxiao.com
     * @param responseType String.class
     * @return
     */
    public <T> HttpEntity<T> get(String url, Class<T> responseType) {
        return exchange(url, HttpMethod.GET, responseType);
    }

    /**
     * @param url          http://api.bianxiao.com
     * @param responseType String.class
     * @return
     */
    public <T> HttpEntity<T> post(String url, Class<T> responseType) {
        return exchange(url, HttpMethod.POST, responseType);
    }

    /**
     * @param url          http://api.bianxiao.com
     * @param method       HttpMethod.POST,HttpMethod.GET {@link HttpMethod}
     * @param responseType String.class
     * @param <T>
     * @return
     */
    public <T> HttpEntity<T> exchange(String url, HttpMethod method, Class<T> responseType) {
        return exchange(url, Maps.newHashMap(), method, responseType);
    }

    /**
     * 支持@RequstBody 传参请求
     *
     * @param url
     * @param headers
     * @param bodyParams
     * @param method
     * @param responseType
     * @param <T>
     * @return
     */
    public <T> HttpEntity<T> exchange(String url, Map<String, String> headers, Object bodyParams, HttpMethod method, Class<T> responseType) {
        return exchange(url, headers, bodyParams, method, responseType, null);
    }

    public HttpEntity<String> exchange(String url, HttpMethod method, Map<String, Object> bodyParams) {
        return exchange(url, bodyParams, method, String.class);
    }

    public <T> HttpEntity<T> exchange(String url, Map<String, Object> bodyParams, HttpMethod method, Class<T> responseType) {
        return exchange(url, Maps.newHashMap(), bodyParams, method, responseType);
    }

    public <T> HttpEntity<T> exchange(String url, Map<String, String> headers, Object bodyParams, HttpMethod method, Class<T> responseType, String contentType) {

        if (MapUtils.isEmpty(headers)) {
            headers = Maps.newHashMap();
        }

        if (StringUtils.isNotBlank(contentType)) {
            headers.put("Content-Type", contentType);
        }
        HttpEntity httpEntity = new HttpEntity<>(bodyParams, RestTemplateUtils.headerParams(headers));
        if (bodyParams instanceof Map) {
            LinkedMultiValueMap multiValueMap = RestTemplateUtils.praseBodyParams((Map<String, Object>) bodyParams);
            httpEntity = new HttpEntity<>(multiValueMap, RestTemplateUtils.headerParams(headers));
            return restTemplate.exchange(url, method, httpEntity, responseType);
        }
        return restTemplate.exchange(url, method, httpEntity, responseType);
    }


}
