package com.bxm.warcar.web.util;

import java.net.URLEncoder;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bxm.warcar.utils.UUIDHelper;

/**
 * 文本加密<br/>
 *
 * @author kerry.jiang
 * @date 2020/10/28 15:03
 */
public class TextEncrypted {

    private static final Logger LOGGER = LoggerFactory.getLogger(TextEncrypted.class);

    private final EncryptorFactory encryptorFactory;

    public TextEncrypted(EncryptorFactory encryptorFactory) {
        this.encryptorFactory = encryptorFactory;
    }

    /**
     * <p>生成带密文的URL</p>
     * <p>对url上的所有参数拼接做加密处理</p>
     *
     * @param cipher 加密方式
     * @param url 原地址
     * @return java.lang.String
     */
    public String genGetUrl(Integer cipher, String url){
        return genGetUrl(cipher, UUIDHelper.generate(), url);
    }

    /**
     * <p>生成带密文的URL</p>
     * <p>对url上的所有参数拼接做加密处理</p>
     *
     * @param cipher 加密方式
     * @param sign 密钥
     * @param url 原地址
     * @return java.lang.String
     */
    public String genGetUrl(Integer cipher, String sign, String url){
        String[] arr = url.split("\\?");
        if(arr.length != 2){
            return url;
        }
        return genGetUrl(cipher, sign, arr[0], arr[1]);
    }

    /**
     * <p>生成带密文的URL</p>
     *
     * @param cipher 加密方式
     * @param sign 密钥
     * @param apiUrl 接口地址
     * @param plaintext 参数明文
     * @return java.lang.String
     */
    public String genGetUrl(Integer cipher, String sign, String apiUrl, String plaintext){
        String ciphertext = encrypt(cipher, sign, plaintext);
        try {
            ciphertext = URLEncoder.encode(ciphertext, "UTF-8");
        }catch (Exception e){
            LOGGER.warn("URLEncoder: ciphertext={},e={}",
                    ciphertext, ExceptionUtils.getStackTrace(e));
        }
        return StringUtils.join(apiUrl, (apiUrl.indexOf('?')>-1?'&':'?'),
                "cipher=",cipher,"&sign=",sign,"&message=",ciphertext);
    }

    /**
     * 加密<br/>
     *
     * @param cipher 加密方式
     * @param sign 密钥
     * @param plaintext 明文
     * @return java.lang.String
     */
    public String encrypt(Integer cipher, String sign, String plaintext){
        String key = StringUtils.reverse(sign);
        Encryptor encryptor = encryptorFactory.get(cipher);
        if(null == encryptor){
            throw new IllegalArgumentException("Not find Encryptor. cipher="+cipher);
        }
        EncryptContext context = new EncryptContext();
        context.setContent(plaintext);
        context.setKey(key);
        return encryptor.encrypt(context);
    }

    /**
     * 解密<br/>
     *
     * @param cipher 加密方式
     * @param sign 密钥
     * @param ciphertext 密文
     * @return java.lang.String
     */
    public String decrypt(Integer cipher, String sign, String ciphertext) throws Exception {
        String key = StringUtils.reverse(sign);
        Encryptor encryptor = encryptorFactory.get(cipher);
        if(null == encryptor){
            throw new IllegalArgumentException("Not find Encryptor. cipher="+cipher);
        }
        EncryptContext context = new EncryptContext();
        context.setContent(ciphertext);
        context.setKey(key);
        return encryptor.decrypt(context);
    }
}
