/*
 * Copyright 2016 big-mouth.cn
 *
 * The Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package com.bxm.warcar.mq.alions;

import com.aliyun.openservices.ons.api.ONSFactory;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.bxm.warcar.mq.*;
import com.bxm.warcar.utils.LifeCycle;

import java.util.Properties;


/**
 * 基于阿里云ONS的消息实现<br>
 * 请参考：<a href="https://www.aliyun.com/product/ons">https://www.aliyun.com/product/ons</a>
 * @author Allen Hu - (big-mouth.cn) 
 * 2016-3-4
 */
public class AlionsProducer extends LifeCycle implements Producer {

    private final Properties properties = new Properties();
    private com.aliyun.openservices.ons.api.Producer producer;
    
    public AlionsProducer(String producerId, String accessKey, String secretKey) {
        super();
        properties.put(PropertyKeyConst.ProducerId, producerId);
        properties.put(PropertyKeyConst.AccessKey, accessKey);
        properties.put(PropertyKeyConst.SecretKey, secretKey);
    }

    public AlionsProducer(Properties properties) {
        this.properties.putAll(properties);
    }

    @Override
    public SendResult send(Message message) {
        com.aliyun.openservices.ons.api.Message msg = new com.aliyun.openservices.ons.api.Message();
        msg.setTopic(message.getTopic());
        msg.setBody(message.getBody());
        msg.setTag(message.getTags());
        int level = message.getDelayTimeLevel();
        if (level > 0)
            msg.setStartDeliverTime(DelayTimeLevel.ofSystemTimeInMillis(level));
        try {
            com.aliyun.openservices.ons.api.SendResult sendResult = producer.send(msg);
            if (null == sendResult)
                throw new SendException();
            return new SendResult(sendResult.getMessageId());
        }
        catch (Exception e) {
            throw new SendException(e);
        }
    }

    @Override
    protected void doInit() {
        producer = ONSFactory.createProducer(properties);

        if (null != producer && producer.isClosed())
            producer.start();
    }

    @Override
    protected void doDestroy() {
        // ons 不能显示的调用 shutdown，因为 ons 客户端添加了关闭的钩子
        // modified by allen at 2018/8/26
//        if (null != producer && producer.isStarted()) {
//            producer.shutdown();
//        }
    }

    @Override
    public void start() {
        init();
    }

    @Override
    public void close() {
        destroy();
    }

    @Override
    public boolean isStarted() {
        if (null == producer) {
            return false;
        }
        return producer.isStarted();
    }
}
