/*
 * Copyright 2016 big-mouth.cn
 *
 * The Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package com.bxm.warcar.mq.alions;

import com.aliyun.openservices.ons.api.Action;
import com.aliyun.openservices.ons.api.ConsumeContext;
import com.aliyun.openservices.ons.api.ONSFactory;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.bxm.warcar.mq.*;
import com.bxm.warcar.utils.LifeCycle;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;

import java.util.Properties;


/**
 * 基于阿里云ONS的消息实现<br>
 * 请参考：<a href="https://www.aliyun.com/product/ons">https://www.aliyun.com/product/ons</a>
 * @author Allen Hu - (big-mouth.cn) 
 * 2016-3-4
 */
public class AlionsConsumer extends LifeCycle implements Consumer {

    private final Properties properties = new Properties();
    private com.aliyun.openservices.ons.api.Consumer consumer;
    private final Listener messageListener;
    private String subExpression;
    
    public AlionsConsumer(String consumerId, String accessKey, String secretKey,
                          Listener messageListener) {
        Preconditions.checkNotNull(messageListener);
        properties.put(PropertyKeyConst.ConsumerId, consumerId);
        properties.put(PropertyKeyConst.AccessKey, accessKey);
        properties.put(PropertyKeyConst.SecretKey, secretKey);

        this.messageListener = messageListener;
    }

    public AlionsConsumer(Properties properties, Listener messageListener) {
        Preconditions.checkNotNull(properties);
        Preconditions.checkNotNull(messageListener);

        this.properties.putAll(properties);
        this.messageListener = messageListener;
    }

    public AlionsConsumer(String consumerId, Properties properties, Listener messageListener) {
        Preconditions.checkArgument(StringUtils.isNotBlank(consumerId));
        Preconditions.checkNotNull(properties);
        Preconditions.checkNotNull(messageListener);

        this.properties.putAll(properties);
        this.properties.put(PropertyKeyConst.ConsumerId, consumerId);
        this.messageListener = messageListener;
    }

    @Override
    protected void doInit() {
        consumer = ONSFactory.createConsumer(properties);

        if (null != consumer) {
            String topic = messageListener.getTopic();
            if (StringUtils.isBlank(topic))
                throw new RuntimeException("topic must has not blank!");
            if (StringUtils.isBlank(subExpression))
                this.subExpression = "*";
            consumer.subscribe(topic, subExpression, new com.aliyun.openservices.ons.api.MessageListener() {
                
                @Override
                public Action consume(com.aliyun.openservices.ons.api.Message message, ConsumeContext context) {
                    Message msg = new Message();
                    msg.setTopic(message.getTopic());
                    msg.setBody(message.getBody());
                    msg.setTags(message.getTag());
                    msg.setKey(message.getKey());
                    msg.setMsgId(message.getMsgID());
                    msg.setReconsumeTimes(message.getReconsumeTimes());

                    ConsumeStatus status = null;
                    if (messageListener instanceof MessageListener) {
                        status = ((MessageListener) messageListener).consume(Lists.newArrayList(msg), context);
                    }
                    else if (messageListener instanceof SingleMessageListener) {
                        status = ((SingleMessageListener) messageListener).consume(msg, context);
                    }
                    if (null == status) {
                        return Action.ReconsumeLater;
                    }

                    switch (status) {
                        case CONSUME_SUCCESS:
                            return Action.CommitMessage;
                        default:
                            return Action.ReconsumeLater;
                    }
                }
            });
            if (consumer.isClosed())
                consumer.start();
        }
    }

    @Override
    protected void doDestroy() {
        shutdown();
    }

    @Override
    public void suspend() {
        if (consumer.isStarted())
            consumer.shutdown();
    }

    @Override
    public void shutdown() {
        // ons 不能显示的调用 shutdown，因为 ons 客户端添加了关闭的钩子
        // modified by allen at 2018/8/26

//        if (consumer.isStarted())
//            consumer.shutdown();
    }

    @Override
    public void start() {
        init();
    }

    @Override
    public boolean isStarted() {
        if (null == consumer) {
            return false;
        }
        return consumer.isStarted();
    }

    public void setSubExpression(String subExpression) {
        this.subExpression = subExpression;
    }

    @Override
    public Listener getMessageListener() {
        return messageListener;
    }
}
