package com.bxm.warcar.integration.pair.local;

import com.bxm.warcar.integration.pair.DefaultValue;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.integration.pair.Value;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * 基于本地缓存的实现
 *
 * @author allen
 * @since 1.0.0
 */
public class LoadingCachePairImpl implements Pair {

    private static final Logger LOGGER = LoggerFactory.getLogger(LoadingCachePairImpl.class);
    private final LoadingCache<String, Optional<String>> cache;

    /**
     * @param pairs 数据来源
     */
    public LoadingCachePairImpl(Pair pairs) {
        this(pairs, 5, TimeUnit.MINUTES);
    }

    /**
     * @param pairs 来源
     * @param duration 刷新间隔时间
     * @param timeUnit 刷新间隔时间单位
     */
    public LoadingCachePairImpl(Pair pairs, int duration, TimeUnit timeUnit) {
        this.cache = CacheBuilder.newBuilder()
                .refreshAfterWrite(duration, timeUnit)
                .expireAfterWrite(duration * 2, timeUnit)
                .build(new CacheLoader<String, Optional<String>>() {
                    @Override
                    public Optional<String> load(String key) throws Exception {
                        Value value = pairs.get(key);
                        return Optional.ofNullable(value.of());
                    }
                });
    }

    @Override
    public Value get(String key) {
        return new DefaultValue(get0(key));
    }

    private String get0(String key) {
        try {
            Optional<String> optional = this.cache.get(key);
            return optional.orElse(null);
        } catch (ExecutionException e) {
            LOGGER.error("get: ", e);
            return null;
        }
    }
}
