package com.bxm.warcar.utils.http;

import com.alibaba.fastjson.JSON;
import com.bxm.warcar.utils.http.pool.HttpClientPool;
import com.google.common.base.Charsets;
import com.google.common.collect.Maps;
import org.apache.commons.collections.MapUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * <h3>基于httpClien  http工具类</h3>
 * <p>
 * 支持 http GET , POST
 * 支持body,header传参
 * 线程池请求{@link HttpClientPool}
 * 其他http请求工具类{@link OkHttpUtils,RestTemplateUtils}
 * </p>
 *
 * @author hcmony
 * @since V1.0.0, 2020/03/16 20:44
 */
public class HttpClientUtils {

    private static CloseableHttpClient httpClient = HttpClients.custom().build();

    /**
     * get 请求url后面拼接参数
     *
     * @param url http://api.bianxiao.com?test=1
     * @return
     */
    public static String get(String url) {
        return get(url, Maps.newHashMap());
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url     http://api.bianxiao.com?test=1
     * @param headers heders传参
     * @return
     */
    public static String get(String url, Map<String, String> headers) {
        return get(url, Maps.newHashMap(), headers);
    }

    /**
     * get 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    public static String get(String url, Map<String, Object> params, Map<String, String> headers) {
        HttpGet httpGet = new HttpGet(OkHttpUtils.appendParams(url, params));
        HttpClientUtils.setHeader(httpGet, headers);
        return doRequest(httpGet, headers);
    }

    /**
     * post 请求
     *
     * @param url http://api.bianxiao.com
     * @return
     */
    public static String post(String url) {
        return post(url, Maps.newHashMap(), Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url    http://api.bianxiao.com
     * @param params params传参
     * @return
     */
    public static String post(String url, Map<String, Object> params) {
        return post(url, params, Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    public static String post(String url, Map<String, Object> params, Map<String, String> headers) {
        HttpPost httpPost = new HttpPost(url);
        HttpClientUtils.setParams(httpPost, params);
        return doRequest(httpPost, headers);
    }

    /**
     * post 请求 支持@RequestBody 传参请求
     *
     * @param url     http://api.bianxiao.com
     * @param o       实体传参
     * @param headers heders传参
     * @return
     */
    public static String postRequestBody(String url, Object o, Map<String, String> headers) {
        HttpPost httpPost = new HttpPost(url);
        HttpClientUtils.setRequestBody(httpPost, o);
        return doRequest(httpPost, headers);
    }


    /**
     * 执行http 请求
     *
     * @param httpRequest httpReq
     * @param headers     请求头
     * @return
     */
    private static String doRequest(HttpRequestBase httpRequest, Map<String, String> headers) {
        CloseableHttpResponse httpResp = null;
        try {
            setHeader(httpRequest, headers);
            httpResp = httpClient.execute(httpRequest);
            return EntityUtils.toString(httpResp.getEntity());
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (httpResp != null) {
                    httpResp.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    /**
     * 设置请求头参数
     *
     * @param httpRequest httpRequest
     * @param headers     请求头参数
     */
    public static void setHeader(HttpRequestBase httpRequest, Map<String, String> headers) {
        if (MapUtils.isNotEmpty(headers)) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                httpRequest.setHeader(entry.getKey(), entry.getValue());
            }
        }
    }

    /**
     * 普通参数请求
     *
     * @param httpPost httpPost
     * @param params   普通参数
     */
    public static void setParams(HttpPost httpPost, Map<String, Object> params) {
        if (MapUtils.isNotEmpty(params)) {
            List<NameValuePair> pairs = new ArrayList<>(params.size());
            params.forEach((key, value) -> pairs.add(new BasicNameValuePair(key, value.toString())));
            httpPost.setEntity(new UrlEncodedFormEntity(pairs, Charsets.UTF_8));
        }
    }

    /**
     * @param httpPost httpPost
     * @param o        @RequestBody 实体
     * @RequestBody 请求
     */
    public static void setRequestBody(HttpPost httpPost, Object o) {
        if (null != o) {
            httpPost.setEntity(new StringEntity(JSON.toJSONString(o), ContentType.APPLICATION_JSON));
        }
    }
}
