package com.bxm.warcar.utils.http;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Maps;
import okhttp3.*;
import org.apache.commons.collections.MapUtils;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;

/**
 * <h3>基于httpClien  http工具类</h3>
 * <p>
 * 支持 http GET , POST , PUT, PATCH , DELETE
 * 支持body,header传参
 * 线程池请求{@link com.bxm.warcar.utils.http.pool.OkHttpPool}
 * 其他http请求工具类{@link HttpClientUtils,RestTemplateUtils}
 * </p>
 *
 * @author hcmony
 * @since V1.0.0, 2019/05/21 09:44
 */
public class OkHttpUtils {
    private static OkHttpClient mOkHttpClient = new OkHttpClient.Builder().build();

    /**
     * get 请求url后面拼接参数
     *
     * @param url http://api.bianxiao.com?test=1
     * @return
     */
    public static String get(String url) throws IOException {
        return packagingResult(getRequest(url));
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url     http://api.bianxiao.com?test=1
     * @param headers heders传参
     * @return
     */
    public static String get(String url, Map<String, String> headers) throws IOException {
        return OkHttpUtils.packagingResult(request(url, HttpMethod.GET, headers));
    }

    /**
     * get 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    public static String get(String url, Map<String, Object> params, Map<String, String> headers) throws IOException {
        return get(OkHttpUtils.appendParams(url, params), headers);
    }

    /**
     * post 请求
     *
     * @param url http://api.bianxiao.com
     * @return
     */
    public String post(String url) throws IOException {
        return post(url, Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url    http://api.bianxiao.com
     * @param params params传参
     * @return
     */
    public static String post(String url, Map<String, Object> params) throws IOException {
        return packagingResult(postRequest(url, buildBody(params)));
    }

    /**
     * post 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    public static String post(String url, Map<String, Object> params, Map<String, String> headers) throws IOException {
        return OkHttpUtils.packagingResult(request(url, HttpMethod.POST, OkHttpUtils.buildBody(params), headers));
    }

    /**
     * post 请求 支持@RequestBody 传参请求
     *
     * @param url     http://api.bianxiao.com
     * @param o       实体传参
     * @param headers heders传参
     * @return
     */
    public static String postRequestBody(String url, Object o, Map<String, String> headers) throws IOException {
        return OkHttpUtils.packagingResult(requestBody(url, HttpMethod.POST, OkHttpUtils.buildRequestBody(o), headers));
    }

    /**
     * @param url
     * @param o   @RequesBody对象
     * @return
     * @throws IOException
     * @RequesBody 传参方式Post请求
     */
    public static Response post(String url, Object o) throws IOException {
        return postRequest(url, buildRequestBody(o));
    }


    /**
     * Post请求
     *
     * @param url
     * @param body 对象
     * @return
     * @throws IOException
     */
    public static Response postRequest(String url, RequestBody body) throws IOException {
        return request(url, HttpMethod.POST, body);
    }

    private static Response getRequest(String url) throws IOException {
        return request(url, HttpMethod.GET);
    }

    /**
     * 支持 http GET , POST , PUT, PATCH , DELETE
     *
     * @param url
     * @param method {@link HttpMethod}
     * @return
     * @throws IOException
     */
    public static Response request(String url, HttpMethod method) throws IOException {
        return request(url, method, Maps.newHashMap());
    }

    public static Response request(String url, HttpMethod method, Map<String, String> headers) throws IOException {
        return request(url, method, null, headers);
    }

    public static Response request(String url, HttpMethod method, RequestBody body) throws IOException {
        return request(url, method, body, null);
    }

    public static Response request(String url, HttpMethod method, RequestBody body, Map<String, String> headers) throws IOException {
        return mOkHttpClient.newCall(setHeaders(buildRequest(url, method, body), headers).build()).execute();
    }

    /**
     * @param url
     * @param method
     * @param body
     * @param headers
     * @return
     * @throws IOException
     * @RequestBody类请求方式
     */
    public static Response requestBody(String url, HttpMethod method, RequestBody body, Map<String, String> headers) throws IOException {
        return mOkHttpClient.newCall(OkHttpUtils.setHeaders(OkHttpUtils.buildRequest(url, method, body), headers).build()).execute();
    }

    public static Request.Builder buildRequest(String url, HttpMethod httpMethod, RequestBody body) {
        if (httpMethod == null) {
            httpMethod = HttpMethod.GET;
        }
        switch (httpMethod) {
            case GET:
                return new Request.Builder().url(url).get();
            case POST:
                return new Request.Builder().url(url).post(body);
            case PUT:
                return new Request.Builder().url(url).put(body);
            case PATCH:
                return new Request.Builder().url(url).patch(body);
            case DELETE:
                return new Request.Builder().url(url).delete(body);
            default: {
                return new Request.Builder().url(url).get();
            }
        }
    }


    /**
     * 结果包装
     *
     * @param response
     * @return
     * @throws IOException
     */
    public static String packagingResult(Response response) throws IOException {
        if (response != null && response.body() != null) {
            return response.body().string();
        }
        return null;
    }

    public static Request.Builder setHeaders(Request.Builder builder, Map<String, String> headers) {
        if (MapUtils.isEmpty(headers)) {
            return builder;
        }
        headers.forEach(builder::header);
        return builder;
    }


    public static String appendParams(String url, Map<String, Object> params) {
        if (MapUtils.isEmpty(params)) {
            return url;
        }
        StringBuffer psb = new StringBuffer();
        params.forEach((key, value) -> {
            try {
                psb.append(URLEncoder.encode(key, "UTF-8"))
                        .append("=")
                        .append(URLEncoder.encode(value.toString(), "UTF-8"))
                        .append("&");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
        });
        if (url.indexOf("?") > 0) {
            return url + "&" + psb.toString();
        }
        return url + "?" + psb.toString();
    }

    public static RequestBody buildBody(Map<String, Object> params) throws IOException {
        FormBody.Builder build = new FormBody.Builder();
        if (MapUtils.isNotEmpty(params)) {
            for (String key : params.keySet()) {
                if (params.get(key) != null) {
                    build.add(URLEncoder.encode(key, "UTF-8"), URLEncoder.encode(params.get(key).toString(), "UTF-8"));
                }
            }
        }
        return build.build();
    }

    /**
     * 使用于@RequestBody参数
     *
     * @param o 请求体
     * @return
     */
    public static RequestBody buildRequestBody(Object o) {
        if (null == o) {
            throw new NullPointerException("okhttp request body is null");
        }
        return RequestBody.create(MediaType.parse("application/json"), JSON.toJSONString(o));
    }

}
