package com.bxm.warcar.utils.http.pool;

import com.bxm.warcar.utils.http.HttpClientUtils;
import com.bxm.warcar.utils.http.HttpPool;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.bxm.warcar.utils.http.properties.HttpProperties;
import com.google.common.collect.Maps;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.util.Map;

/**
 * <h3>类的基本描述</h3>
 * <p> 单个请求 {@link HttpClientUtils}</p>
 * @author hcmony
 * @since V1.0.0, 2019/12/30 16:39
 */
public class HttpClientPool implements HttpPool {
    private final CloseableHttpClient httpClient;
    private final RequestConfig requestConfig;

    public HttpClientPool(HttpProperties properties) {
        ConnectionSocketFactory plainsf = PlainConnectionSocketFactory.getSocketFactory();
        LayeredConnectionSocketFactory sslsf = SSLConnectionSocketFactory.getSocketFactory();

        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", plainsf)
                .register("https", sslsf)
                .build();

        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(registry);
        connManager.setMaxTotal(properties.getMaxTotal());
        connManager.setDefaultMaxPerRoute(properties.getDefaultMaxPerRoute());

        this.requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(properties.getConnectionRequestTimeout())
                .setConnectTimeout(properties.getConnectTimeout())
                .setSocketTimeout(properties.getSocketTimeout())
                .setExpectContinueEnabled(false)
                .build();
        this.httpClient = HttpClients.custom()
                .setConnectionManager(connManager)
                .build();
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url http://api.bianxiao.com?test=1
     * @return
     */
    @Override
    public String get(String url) {
        return get(url, Maps.newHashMap());
    }

    /**
     * get 请求url后面拼接参数
     *
     * @param url     http://api.bianxiao.com?test=1
     * @param headers heders传参
     * @return
     */
    @Override
    public String get(String url, Map<String, String> headers) {
        return get(url, Maps.newHashMap(), headers);
    }

    /**
     * get 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String get(String url, Map<String, Object> params, Map<String, String> headers) {
        HttpGet httpGet = new HttpGet(OkHttpUtils.appendParams(url, params));
        HttpClientUtils.setHeader(httpGet, headers);
        return doRequest(httpGet, headers);
    }

    /**
     * post 请求
     *
     * @param url http://api.bianxiao.com
     * @return
     */
    @Override
    public String post(String url) {
        return post(url, Maps.newHashMap(), Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url    http://api.bianxiao.com
     * @param params params传参
     * @return
     */
    @Override
    public String post(String url, Map<String, Object> params) {
        return post(url, params, Maps.newHashMap());
    }

    /**
     * post 请求
     *
     * @param url     http://api.bianxiao.com
     * @param params  params传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String post(String url, Map<String, Object> params, Map<String, String> headers) {
        HttpPost httpPost = new HttpPost(url);
        HttpClientUtils.setParams(httpPost, params);
        return doRequest(httpPost, headers);
    }

    /**
     * post 请求 支持@RequestBody 传参请求
     *
     * @param url     http://api.bianxiao.com
     * @param o       实体传参
     * @param headers heders传参
     * @return
     */
    @Override
    public String postRequestBody(String url, Object o, Map<String, String> headers) {
        HttpPost httpPost = new HttpPost(url);
        HttpClientUtils.setRequestBody(httpPost, o);
        return doRequest(httpPost, headers);
    }


    private String doRequest(HttpRequestBase httpReq, Map<String, String> headers) {
        CloseableHttpResponse httpResp = null;
        try {
            httpReq.setConfig(requestConfig);
            HttpClientUtils.setHeader(httpReq, headers);
            httpResp = httpClient.execute(httpReq);
            return EntityUtils.toString(httpResp.getEntity());
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (httpResp != null) {
                    httpResp.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return null;
    }
}
