package com.bxm.warcar.utils.device;

import com.bxm.warcar.utils.crypto.Base64UrlSafeStringAesCipher;
import com.bxm.warcar.utils.crypto.HexAesCipher;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import java.util.Optional;

/**
 * <p>简单版的 Device：移动设备通用设备标识。</p>
 *
 * <p>使用 {@link Builder} 来构造一个 {@link SimpleDevice} 对象</p>
 * <p>使用 {@link Base64Cipher} 可以对{@link SimpleDevice}实例进行加解密，加密后可得到一个URLSafeString的密文字符串。</p>
 *
 * @author allen
 * @date 2021-08-26
 * @since 1.0
 * @see Builder
 * @see Base64Cipher
 */
public class SimpleDevice {

    private static final String SPLIT = "|";

    public static final SimpleDevice EMPTY = new SimpleDevice();

    private final String text;

    /**
     * imei
     */
    private String imei;
    /**
     * imei md5
     */
    private String imeiMd5;
    /**
     * 安卓id
     */
    private String androidid;
    /**
     * 安卓id md5
     */
    private String androididMd5;
    /**
     * 安全联盟定义的用户唯一标识OAID
     */
    private String oaid;
    /**
     * idfa
     */
    private String idfa;
    /**
     * idfa md5
     */
    private String idfaMd5;
    /**
     * oaid md5
     */
    private String oaidMd5;
    /**
     * Google android ID
     */
    private String gaid;
    /**
     * Google android ID md5
     */
    private String gaidMd5;

    private SimpleDevice() {
        this(null);
    }

    private SimpleDevice(String text) {
        this.text = text;
        String[] strings = Optional.ofNullable(StringUtils.splitPreserveAllTokens(text, SPLIT)).orElse(new String[0]);
        for (int i = 0; i < strings.length; i++) {
            if (i == 0) {
                this.imei = strings[i];
            } else if (i == 1) {
                this.imeiMd5 = strings[i];
            } else if (i == 2) {
                this.androidid = strings[i];
            } else if (i == 3) {
                this.androididMd5 = strings[i];
            } else if (i == 4) {
                this.oaid = strings[i];
            } else if (i == 5) {
                this.idfa = strings[i];
            } else if (i == 6) {
                this.idfaMd5 = strings[i];
            } else if (i == 7) {
                this.oaidMd5 = strings[i];
            } else if (i == 8) {
                this.gaid = strings[i];
            } else if (i == 9) {
                this.gaidMd5 = strings[i];
            }
        }
    }

    @Override
    public String toString() {
        return text;
    }

    public String getImei() {
        return imei;
    }

    public String getImeiMd5() {
        return imeiMd5;
    }

    public String getAndroidid() {
        return androidid;
    }

    public String getAndroididMd5() {
        return androididMd5;
    }

    public String getOaid() {
        return oaid;
    }

    public String getIdfa() {
        return idfa;
    }

    public String getIdfaMd5() {
        return idfaMd5;
    }

    public String getOaidMd5() {
        return oaidMd5;
    }

    public String getGaid() {
        return gaid;
    }

    public String getGaidMd5() {
        return gaidMd5;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        /**
         * imei
         */
        private String imei;
        /**
         * imei md5
         */
        private String imeiMd5;
        /**
         * 安卓id
         */
        private String androidid;
        /**
         * 安卓id md5
         */
        private String androididMd5;
        /**
         * 安全联盟定义的用户唯一标识OAID
         */
        private String oaid;
        /**
         * idfa
         */
        private String idfa;
        /**
         * idfa md5
         */
        private String idfaMd5;
        /**
         * oaid md5
         */
        private String oaidMd5;
        /**
         * Google android ID
         */
        private String gaid;
        /**
         * Google android ID md5
         */
        private String gaidMd5;

        public Builder imei(String imei) {
            this.imei = imei;
            return this;
        }

        public Builder imeiMd5(String imeiMd5) {
            this.imeiMd5 = imeiMd5;
            return this;
        }

        public Builder androidid(String anid) {
            this.androidid = anid;
            return this;
        }

        public Builder androididMd5(String anidMd5) {
            this.androididMd5 = anidMd5;
            return this;
        }

        public Builder oaid(String oaid) {
            this.oaid = oaid;
            return this;
        }

        public Builder idfa(String idfa) {
            this.idfa = idfa;
            return this;
        }

        public Builder idfaMd5(String idfaMd5) {
            this.idfaMd5 = idfaMd5;
            return this;
        }

        public Builder oaidMd5(String oaidMd5) {
            this.oaidMd5 = oaidMd5;
            return this;
        }

        public Builder gaid(String gid) {
            this.gaid = gid;
            return this;
        }

        public Builder gaidMd5(String gidMd5) {
            this.gaidMd5 = gidMd5;
            return this;
        }

        public SimpleDevice build() {
            String s = append(this.imei, this.imeiMd5, this.androidid, this.androididMd5, this.oaid, this.idfa, this.idfaMd5,
                    this.oaidMd5, this.gaid, this.gaidMd5);
            return new SimpleDevice(s);
        }

        private String append(String...strings) {
            StringBuilder s = new StringBuilder();
            for (String string : strings) {
                if (StringUtils.isBlank(string)) {
                    string = StringUtils.EMPTY;
                }
                s.append(string).append(SPLIT);
            }
            return StringUtils.removeEnd(s.toString(), SPLIT);
        }
    }

    @Slf4j
    public static class Base64Cipher {

        private static final String KEY = "bXm2021#world.bs";

        public static String encrypt(SimpleDevice device) {
            return encrypt(KEY, device);
        }

        public static String encrypt(String key, SimpleDevice device) {
            try {
                return Base64UrlSafeStringAesCipher.encrypt(key, device.toString());
            } catch (Exception e) {
                log.error("encrypt: ", e);
                return null;
            }
        }

        public static SimpleDevice decrypt(String secretText) {
            return decrypt(KEY, secretText);
        }

        public static SimpleDevice decrypt(String key, String secretText) {
            try {
                String plaintext = Base64UrlSafeStringAesCipher.decrypt(key, secretText);
                return new SimpleDevice(plaintext);
            } catch (Exception e) {
                log.error("decrypt: ", e);
                return SimpleDevice.EMPTY;
            }
        }
    }

    @Slf4j
    public static class HexCipher {

        private static final String KEY = "bXm2021#world.hX";

        public static String encrypt(SimpleDevice device) {
            return encrypt(KEY, device);
        }

        public static String encrypt(String key, SimpleDevice device) {
            try {
                return HexAesCipher.encrypt(key, device.toString());
            } catch (Exception e) {
                log.error("encrypt: ", e);
                return null;
            }
        }

        public static SimpleDevice decrypt(String secretText) {
            return decrypt(KEY, secretText);
        }

        public static SimpleDevice decrypt(String key, String secretText) {
            try {
                String plaintext = HexAesCipher.decrypt(key, secretText);
                return new SimpleDevice(plaintext);
            } catch (Exception e) {
                log.error("decrypt: ", e);
                return SimpleDevice.EMPTY;
            }
        }
    }
}
