package com.omni.ads.container;

import com.google.gson.JsonSyntaxException;
import com.google.inject.Inject;
import com.omni.ads.baseconfig.ApiContainer;
import com.omni.ads.baseconfig.ApiException;
import com.omni.ads.baseconfig.Constant;
import com.omni.ads.baseconfig.ResultDto;
import com.omni.ads.api.AdsCreativityApi;
import com.omni.ads.exception.OmniAdsResponseException;
import com.omni.ads.model.adscreativity.*;
import com.omni.ads.model.adscreativity.vo.HttpPage;
import com.omni.ads.model.adsgroup.ValidationGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.groups.Default;
import java.util.List;
import java.util.Set;

public class AdsCreativityApiContainer extends ApiContainer {
    @Inject
    AdsCreativityApi api;
    private static final Logger LOGGER = LoggerFactory.getLogger(AdsCreativityApiContainer.class);

    /**
     * 获取单个广告创意详情
     *
     * @param adId (required)
     * @return AdsAdCollectResp
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<AdsAdCollectResp> v3AdGet(@RequestParam(required = true) Long adId) throws ApiException, OmniAdsResponseException {
        ResultDto<AdsAdCollectResp> rt = new ResultDto<>();
        try {
            AdsGetCreativityResponse resp = api.getAdsCreativity(adId);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3AdGet exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return rt;
        } catch (JsonSyntaxException e1) {
            return rt;
        }

    }

    /**
     * 获取广告组下广告详情列表,营销平台批量编辑广告组下所有广告创意时调用
     *
     * @param data (required)
     * @return List<AdsAdCollectResp>
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsAdCollectResp>> v3AdList(@ModelAttribute AdsAdCollectListReq data) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsAdCollectResp>> rt = new ResultDto<>();
        try {
            AdsGetCreativityListOfAdResponse resp = api.getAdsCreativityListOfAd(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3AdList exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return rt;
        }
    }

    /**
     * 分页获取广告创意列表
     *
     * @param data (required)
     * @return HttpPage<AdsAdDataResp>
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<HttpPage<AdsAdDataResp>> v3AdPage(@Validated(value = {ValidationGroup.Find.class, Default.class})
                                                       @ModelAttribute AdsAdListForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<HttpPage<AdsAdDataResp>> rt = new ResultDto<>();
        try {
            AdsGetListByPageResponse resp = api.getAdsListByPage(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3AdPage exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return rt;
        }
    }

    /**
     * 批量启停广告创意
     *
     * @param adForm (required)
     * @return List<AdsAdUpdateInfo>
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsAdUpdateInfo>> v3AdStatus(@Validated(value = ValidationGroup.Status.class)
                                                       @ModelAttribute AdsAdStatusForm adForm) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsAdUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsBatchEditStatusResponse resp = api.batchEditAdStatus(adForm);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3AdStatus exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return rt;
        }
    }


    /**
     * 批量操作广告创意
     * 批量操作(新增/编辑/删除)广告创意(新增和编辑限制同一个广告组下，删除不限制，可操作1-50个)
     *
     * @param data (required)
     * @return Set<Long>
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<Set<Long>> v3AdBatch(AdsAdCollectBatchReq data) throws ApiException, OmniAdsResponseException {
        ResultDto<Set<Long>> rt = new ResultDto<>();
        try {
            AdsBatchOprateAdsResponse resp = api.batchOprateAds(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3AdBatch exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return rt;
        }
    }

    /**
     * 批量复制广告创意
     * 批量复制广告创意(目标广告组必须和当前所选创意对应广告组的推广目标、流量场景、应用、快应用、落地页一致)
     *
     * @param data (required)
     * @return HttpResponse
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<Void> v3AdBatchReplicate(@Validated(value = {ValidationGroup.BatchReplicate.class})
                                        @ModelAttribute AdsAdReplicateForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<Void> rt = new ResultDto<>();
        try {
            AdsBatchReplicateResponse response = api.batchReplicateAds(data);
            rt.setRet(response.getCode());
            rt.setMsg(response.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3AdBatchReplicate exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return rt;
        }
    }

}

